(ns codescene.features.config.presets
  (:require [clojure.java.jdbc :as jdbc]
            [codescene.features.config.properties :as p]
            [codescene.features.util.coll :refer [do-diff]]
            [codescene.features.config.presets-db :as db]
            [codescene.features.delta.code-health.ch-config :as ch-config]
            [codescene.features.delta.coverage.cov-config :as cov-config]
            [codescene.features.delta.integration.integration-config :as int-config]
            [medley.core :as m]))

(defmulti properties (fn [{:keys [category]}] category))
;; needs to be fixed
(defmulti find-uses (fn [{:keys [category]} _db-spec] category))

(defmethod find-uses :default
  [_ _] [])

(defn get-settings [db-spec id scope]
  (jdbc/with-db-transaction
    [tx db-spec]
    (when-let [main (db/get-settings tx id scope)]
      (assoc main
        :uses (find-uses main tx)
        :kv (p/from-db (properties main) (db/get-settings-kv tx id))))))

(def delete-settings db/delete-settings)

(defn params->bool [s]
  (case s
    nil nil
    ("true" true "1") true
    ("false" false "0") false))

(defn create-settings [{:keys [db-spec scope encryptor params html-form-input?]}]
  (jdbc/with-db-transaction
    [tx db-spec]
    (let [base {:category (:category params)
                :label (:label params)}
          id (db/insert-settings tx (merge base scope))]
      (doseq [[k v] (p/from-req-params (properties params) params encryptor html-form-input?)]
        (db/insert-settings-kv tx id k v))
      id)))

(def list-settings db/list-settings)

(defn available-settings [db-spec scope types]
  (->> (db/active-settings-by-categories db-spec types scope)
       (group-by :category)))

(defn settings-ids-by-category
  "Returns list of {:id, :active? } of settings by category"
  [db-spec category]
  (db/get-settings-ids-by-category db-spec category))

(defn update-settings [db-spec scope encryptor {:keys [id setting-id label active] :as params}]
  (jdbc/with-db-transaction
    [tx db-spec]
    (let [id (or id setting-id)
          base (m/assoc-some (db/get-settings tx id scope)
                             :label label
                             :active? (params->bool active))
          data (p/from-req-params (properties params) params encryptor true)
          {:keys [removed added updated]} (do-diff first (db/get-settings-kv tx id) data)]
      (db/update-settings tx base)
      (doseq [[k _] removed]
        (db/delete-settings-kv tx id k))
      (doseq [[k v] added]
        (db/insert-settings-kv tx id k v))
      (doseq [[_old new] updated]
        (db/update-settings-kv tx id (first new) (second new)))
      id)))

(defn invert-active [db-spec id]
  (jdbc/with-db-transaction
    [tx db-spec]
    (when-let [main (db/get-settings tx id {})]
      (db/update-settings tx (update main :active? not)))))

(defmethod properties "pr-quality-gates"
  [params]
  (mapv (fn [{:keys [id] :as p}]
          (if (= id :qg-preset) (assoc p :default "minimal") p))
        ch-config/qg-properties))

(defmethod properties "cov-config"
  [params]
  (mapv (fn [{:keys [id] :as p}]
          (if (= id :qg-preset) (assoc p :default "custom") p))
        (concat cov-config/settings-properties int-config/general-delta-properties)))

(defn new-properties [category]
  (p/from-db (properties {:category category}) {}))