(ns codescene.features.client.graphql
  "Quick helper for writing graphql queries. This does not, in fact, fulfil the GraphQL spec completely."
  (:require [clojure.string :as str]
            [clojure.walk :as walk])
  (:import (clojure.lang Keyword Symbol Sequential Cons Named IPersistentList)
           (java.util Map Map$Entry)))

(defn as
  "Convenience function to generate a symbol with meta that represents an aliased name in GraphQL. Useful
  when programmatically generating these values."
  [alias-name prop-name]
  (with-meta
    (symbol (name alias-name))
    {(keyword (name prop-name)) true}))

(defn- seq-param?
  "Internal function. Returns true if param is sequeable but not a string."
  [it]
  (and (seqable? it) (not (string? it))))

(defn- valid-list? [l]
  (and (instance? Named (first l))
       (every? (complement seq-param?) (butlast l))
       true))

(defn- update-arguments*
  "Updates argument in list lst with (fn [op args])"
  [lst f]
  (let [op (first lst)
        traverse (fn x [[it & more]]
                   (if more
                     (conj (x more) it)
                     (list (if (seq-param? it)
                             (f op it) it))))]
    (traverse lst)))

(defn update-arguments
  [f v]
  (if (and (or (instance? Cons v) (list? v))
           (valid-list? v))
    (update-arguments* v f)
    (walk/walk (partial update-arguments f) identity v)))

(defn- name*
  "Like clojure.core/name, but it will prepend @ if name starts with %.
   This extension is due to GraphQL directives using @ prefix and that has special meaning in Clojure."
  [x]
  (let [n (name x)]
    (if (str/starts-with? n "%")
      (str \@ (subs n 1))
      n)))

(defmulti ^:private q-print "Print a value into the query form." class)

(defmethod q-print :default [x] (pr-str x))
(defmethod q-print nil [x] "null")
(defmethod q-print Keyword [x] (name x))
(defmethod q-print Symbol [x]
  (if-let [alias-kw (-> x meta (dissoc :line) ffirst)]
    (str (name* x) ": " (name alias-kw))
    (name* x)))

(defmethod q-print Map$Entry [x] (str (q-print (key x)) ": " (q-print (val x))))
(defmethod q-print Map [x]
  (str \{ (str/join ", " (sort (map q-print x))) \}))
(defmethod q-print Sequential [x] (str \[ (str/join ", " (map q-print x)) \]))

(prefer-method q-print Map$Entry Sequential)

(defn- print-list [x]
  (or (valid-list? x)
      (throw (Exception. (str "Invalid GraphQl List operator " (pr-str x)))))
  (reduce
    (fn [s v]
      (if (seq-param? v)
        (cond-> s
          (seq v) (str \( (str/join ", " (sort (map q-print v))) \)))
        (str s \space (q-print v))))
    (q-print (first x))
    (rest x)))

(defmulti ^:private q-print-lines "Prints a value into a sequence of lines" class)
(defmethod q-print-lines :default [x] [(q-print x)])
(defmethod q-print-lines Map$Entry [x]
  (concat (q-print-lines (key x))
          (when-some [v (val x)] (q-print-lines v))))
(defmethod q-print-lines Map [x]
  (let [per-entry (sort-by #(some identity %) (map q-print-lines x))]
    (flatten ["{" (keep #(some->> % (str "  ")) (flatten per-entry)) "}"])))
(defmethod q-print-lines Sequential [x]
  (flatten ["{" (->> x
                     (mapcat #(if (map? %) % [%]))
                     (mapcat q-print-lines)
                     (keep #(some->> % (str "  ")))) "}"]))
(defmethod q-print-lines IPersistentList [x] [(print-list x)])
(defmethod q-print-lines Cons [x] [(print-list x)])

(prefer-method q-print-lines Cons Sequential)
(prefer-method q-print-lines IPersistentList Sequential)
(prefer-method q-print-lines Map$Entry Sequential)

(defn- collapse-open-parenthesis
  "Collapses open curly parentheses that are on their own line into the previous line.

  First line is not affected"
  [lines]
  (let [open-parenthesis? #(and % (re-find #"^\s*\{$" %))
        reduced-lines (for [[line next-line] (take-while some? (iterate next lines))
                            :when (not (open-parenthesis? line))]
                        (cond-> line (open-parenthesis? next-line) (str " {")))]
    (if (open-parenthesis? (first lines))
      (concat [(first lines)] reduced-lines)
      reduced-lines)))

(defn generate-query
  "Generate a GraphQL query. The query is formatted for your convenience.

  keywords and symbols can be used interchangeably in queries. If namespaced, the namespace is ignored,
  which enables that you use `(...) for splicing values into queries

  Query syntax comes down to:
  item or collection selector { projection }

  If top-level coll is a map, the entries are treated as a separate element of toplevel each:
  '{mutation [:a] query [:b]} -> mutation { a }\n query { b }

  If that is not desired wrap the top level with a vector:

  '[{mutation [:a]}] -> { mutation { a }}

  When the key position needs to have multiple terms use LIST to keep them together:

  '{(query noFragments) [:a]} -> query noFragments { a }

  Param lists:

  If last element in list is a map, the map is emitted as a param list for symbol before it, values like
  vectors are treated differently inside the param lists:

  - '{(user {:id 4}) [:a]} -> user(id: 4) { a }
  - '{(profiles {:handles [\"zuck\" \"coca-cola\"]}) [:a]} -> profiles(handles: [\"zuck\", \"coca-cola\"]) { a }

  Aliasing:

  If element has a meta, then meta is printed as the element and the element printed as an alias.

  '{(^:profilePic smallPic {:size 64}) [:a]} -> smallPic: profilePic(size: 64) { a }

  Special character:

  Character @ cannot be in clojure symbols or keywords, so any symbols or keywords
  starting with % are emitted starting with @.

  Projection using a map

  When specifying map as the projection, its keys are emitted into the projection and values are as sub-levels,
  nil meaning no sub-level:

  {:id nil :sub ...} produces {id sub { ... }} projection

  Projection using vectors:

  You can use vectors to simplify. Keywords/symbols in vector are emitted as is, and maps are emitted
  as described previously for map. You can have multiple maps in the vector, and it's the same as if you had
  one merged map. The only exception to this rule is when you put a vector into
  the vector. In that case it's equivalent to being paired with previous element in a map.

  [[:a :b {:c [:d] :e nil} :f [:g]]] produces { a b c { d } e f { g } }"
  [query-data]
  (->> (mapcat q-print-lines query-data)
       collapse-open-parenthesis
       (str/join \newline)))
