grammar Kotlin;

import KotlinSharedRulesLexer;

translationunit
:
	expression* EOF
;

expression: declarations_without_bodies
          | guarded_function_expression
          | function_expression
          | function_declaration
          | init_function
          | constructor_function
          | data_class
          | class_block
          | class_block_statement
          | property
          | spek_group
          | spek_test
          | anything;

anything : ~(LeftBrace | RightBrace);

// Need to match and parse away expect functions since they don't have a body:
declarations_without_bodies: ('expect' | 'abstract') FUNCTION
                           | 'annotation' (CLASS | OBJECT);

class_block: class_container class_name anything*? LeftBrace expression*? RightBrace;

class_container: (CLASS | OBJECT);

class_name: ID;

data_class: DATA CLASS expression;

property: (VAL | VAR) property_name;
property_name: ID
             | ID extension_property_name;
extension_property_name: '.' property_name;

init_function: INIT LeftBrace function_body RightBrace;
constructor_function: CONSTRUCTOR function_scope;

class_block_statement : LeftBrace expression*? RightBrace;

spek_group: ('group' | 'given') LeftParen group_name RightParen LeftBrace expression*? RightBrace;
group_name: LITERAL | BACKTICKED;
spek_test: 'it' LeftParen spek_test_name RightParen LeftBrace function_body RightBrace;
spek_test_name: LITERAL | BACKTICKED;

function_declaration : FUNCTION generic_modifier? function_name function_scope;

function_scope: function_signature full_body;

// We don't treat function expressions in general as 1st class functions (see comment below).
// However, a function expression guarded by `when` is a common idiom. Let's make sure we
// capture that.
// Example:
//
//  fun CursorType.toJsCursorType(): String = when (this) {
guarded_function_expression: FUNCTION generic_modifier? function_name function_signature '=' WHEN guarded_expression_content_list full_body;

guarded_expression_content: ~(FUNCTION | CLASS | OBJECT | LeftBrace | RightBrace |
                            LeftParen | RightParen |
                           'private' | 'public' | 'override' | 'interface' |
                           'final' | 'open' | 'actual' |
                           'expect' | 'abstract' | 'annotation');

guarded_expression_content_list: guarded_expression_content
                               | LeftParen guarded_expression_content+ RightParen;

// Introduce a separate rule for single expression functions. Note that they can still
// span many lines of code (multi-line lambdas, object creation, etc.) so important to parse out.
function_expression: FUNCTION generic_modifier? function_name function_signature expression_body;

generic_modifier: generic_signature '.'?;
generic_signature: GENERIC_START generic_part* GENERIC_END;
generic_part: ~(LeftBrace | RightBrace | GENERIC_START | GENERIC_END)
            | generic_signature;

function_name: ID
             | BACKTICKED
             | ID extension_function
             | ID generic_modifier ID; // E.g. fun <T, R> Collection<T>.fold(...)

extension_function: '.' function_name;

full_body: LeftBrace function_body RightBrace;
// NOTE: we don't treat single-line expression as regular functions: they won't contribute
// to lower code health, but we need their name for LCOM4 calculations..
expression_body: '=' (object_creation_expression
                      | function_call_expression
                      |  single_line_expression);
single_line_expression: .;
object_creation_expression: object_name object_creation_arguments;

object_name: ID;
object_creation_arguments: LeftParen object_creation_arguments_list*? RightParen;
object_creation_arguments_list: ~(LeftParen | RightParen)
                             | object_creation_arguments;

function_call_expression: function_call;
function_call: function_name function_call_arguments
               ('.' function_name function_call_arguments)*; // chained function calls like repository.getCustomerCredit().map { response ->
function_call_arguments: function_call_trailing_lambda_argument
                        | function_call_arguments_list;
function_call_arguments_list: object_creation_arguments;
function_call_trailing_lambda_argument: block_statement; // e.g. map { response -> ... }

function_signature: LeftParen function_definition_params_list? RightParen return_type?;

return_type: COLON ~(LeftBrace | '=')+;

function_definition_params_list : function_param
                                | function_definition_params_list ',' function_param
                                ;

//function_param: param_part function_param*?
//              | LeftParen function_param*? RightParen;
function_param: parameter_modifier? parameter_name ':' parameter_type default_parameter_value?;
parameter_modifier: 'vararg';
parameter_name: ID | CONSTRUCTOR | INIT;

parameter_type: ID nullable_parameter?
              | generic_parameter_type nullable_parameter?
              | labmda_argument
              | function_argument_signature
              | wildcard
              | nested_parameter_type;

nested_parameter_type: ID '.' parameter_type;

wildcard: ASTERIX;
nullable_parameter: '?';

generic_parameter_type: ID GENERIC_START generic_arg_type_list GENERIC_END;
generic_arg_type_list : generic_type_modifiers? parameter_type
                      | generic_arg_type_list ',' generic_type_modifiers? parameter_type
                      ;
generic_type_modifiers: 'in' | 'out';

labmda_argument: extension_argument_name? LeftParen function_definition_params_list? RightParen '->' lambda_return_type;
extension_argument_name: ID '.' ID?;
lambda_return_type: parameter_type;

function_argument_signature: LeftParen type_arguments_params_list? RightParen '->' lambda_return_type;
type_arguments_params_list : parameter_type
                           | type_arguments_params_list ',' parameter_type
                           ;

default_parameter_value: '=' default_paramter_value_specifier;
default_paramter_value_specifier : block_statement | default_init_with_function_call | nested_value_access | .;
nested_value_access: ID | nested_value_access '.' ID;
default_init_with_function_call: nested_value_access LeftParen type_arguments_params_list? RightParen;

function_body : function_body_statement*?;

function_body_statement : block_statement
                        | anything;

block_statement : LeftBrace function_body_statement*? RightBrace;