(ns codescene.features.recommendations.definitions.recommendations
  (:require
   [codescene.features.recommendations.definitions.predicates :refer [predicates]]
   [codescene.features.recommendations.definitions.recommendation-categories :refer [recommendation-categories]]))

(def stopdcl
  {:title "Stop the code health decline"
   :id :stopdcl
   :resolved :stopdcl-ok
   :creation-at :analysis-complete
   :category (:manage-risk recommendation-categories)
   :activate {:prerequisites {:required #{}}
              :inputs [:goals :analysis]
              :criteria [{:init-test :unhandled-notification-files
                          :params {:notification-type :declining-hotspots}}]}
   :issue-formats {:multiple "%d files are declining in Code Health, meaning they are now more expensive to maintain than they used to be."
                   :singular "A file is declining in Code Halth, meaning that it is becoming more expensive to maintain."}
   :opportunity-paragraphs ["No matter the baseline, you never want your code to get worse and to take on additional technical debt."
                            "By implementing “Quality Gates” for Hotspots you can improve and maintain a high level of Code Health."]
   :scope :project
   :per-item-completion-tests [{:test (:goal-set? predicates)
                                ;; These are params that we add here to
                                ;; the test. Other params can be added
                                ;; later.  We also would
                                ;; accept :planned-refactoring, even
                                ;; though the focus
                                ;; is :ensure-code-health
                                :params {:goal-types #{:ensure-code-health :planned-refactoring}}}
                               {:test (:no-goal-violation? predicates)
                                ;; "duration" refers to how long the rec has been active
                                :min-duration-days 28}

                               ;; TODO: right approach?
                               {:test (:file-removed? predicates)
                                :short-circuit true}]})

(def confpr
  {:title "Enable automated code review"
   :id :confpr
   :resolved :confpr-ok
   :creation-at :project-creation
   :category (:technical-debt recommendation-categories)
   :activate {:prerequisites {:required #{:vizhots}
                              :required-one-of [:worst :stopdcl]}
              ;; in this case, we always initialize at project creation,
              ;; but :config would be necessary for initializing existing
              ;; projects (ie. don't initialize if PRs are configured already
              :inputs [:config]
              :criteria [{:not :prs-configured?}]}
   :issue-formats {:singular "PR integrations are not configured for this project."}
   :opportunity-paragraphs ["No matter what the situation is, the top priority should be to stop things from getting worse."
                            "CodeScene's PR integration prevents technical debt from building up by giving feedback to developers that they can act on <em>before</em> their code reaches the main branch."]
   :scope :project
   :completion-tests [{:test (:prs-configured? predicates)}
                      {:test (:prs-received? predicates)}]}) 


(def vizhots
  {:title "Visualize the development hotspots to become familiar with the codebase"
   :id :vizhots
   :resolved :vizhots-ok
   :creation-at :project-creation
   :category (:learn-codescene recommendation-categories)
   :activate {:prerequisites {:required #{}}
              :inputs [:user-acts]
              :criteria [:completion]}
   :issue-formats {}
   :opportunity-paragraphs ["The hotspot is a key concept in CodeScene. Most development activity occurs in a relatively small number of files."
                            "By visualizing those files, you will gain a better understanding of work patterns inside the project, as well as how CodeScene can help you prioritize improvements."]
   :scope :user
   :completion-tests [{:test (:user-has-seen-hotspots? predicates)}]})


(def worst
  {:title "Inspect the Worst Performer for long-term risk"
   :id :worst
   :resolved :worst-ok
   :creation-at :analysis-complete
   :category (:manage-risk recommendation-categories)
   :activate {:prerequisites {:required #{:vizhots}}
              :inputs [:analysis :goals]
              :criteria [{:not (:goal-set? predicates)
                          :params  {:goal-types #{:ensure-code-health :planned-refactoring}}}]}
   :issue-formats {:singular "No goal is set on the worst performing file in the project."}
   :opportunity-paragraphs ["The <q>worst performer</q> in a project is the file with the lowest Code Health. This may or may not be important, depending on whether the file is a hotspot or will be part of upcoming changes."
                            "Awareness of files like this one can allow you to plan accordingly. Setting a quality gate on the file will help prevent a future decline in a potentially problematic file."]
   :scope :project
   :per-item-completion-tests [{:test (:goal-set? predicates)
                                :params {:goal-types #{:ensure-code-health :no-problem :planned-refactoring}}}]})

(def fixtd
  {:title "Remediate high-interest technical debt"
   :id :fixtd
   :resolved :fixtd-ok
   :category (:manage-risk recommendation-categories)
   :activate {}
   :issue-formats {:singular "The code health of these files has not improved."}
   :opportunity-paragraphs ["These files are hotspots that have low Code Health"]
   :user-scope :all-project-users
   :per-item-completion-tests [{:test (:goal-set? predicates)
                                :params {:goal-types #{:planned-refactoring}}
                                ;; TODO: think about this some
                                ;; more. We need a way to express
                                ;; this: all items succeed if one
                                ;; criteria is met. Similar logic could be used for :not 
                                :modifier :one-of}
                               {:test (:code-health-improved? predicates)
                                ;; TODO: this might work in this case,
                                ;; but it's a bit weak. The idea here
                                ;; is that one file has a goal AND has
                                ;; improved. To really express that,
                                ;; the `:one-of` idea should be
                                ;; *above* these 2 predicates.
                                :modifier :one-of}]
   })

(def all
  {:confpr confpr
   :stopdcl stopdcl
   :vizhots vizhots
   :worst worst
   :fixtd fixtd})

(defn- completion-tests-for
  [recommendation]
  (concat (get recommendation :completion-tests [])
          (get recommendation :per-item-completion-tests [])))

(defn related-events-as-predicate-templates
  "Given a recommendation definiton, returns a list of predicate
  templates (see the `map-as-predicate` function). These predicates
  correspond to all the events that are related to the recommendation:
  resolved and unresolved events for the recommendation itself, as
  well as resolved and unresolved events for all the completion tests."
  [user-id {:keys [id completion-tests per-item-completion-tests resolved] :as _recommendation}]
  (let [predicate-events (->> (concat (or completion-tests []) (or per-item-completion-tests []))
                              (map :test)
                              (mapcat
                                (fn [{:keys [resolved unresolved stream]}]
                                  (->> [resolved unresolved]
                                       (remove nil?)
                                       (map (fn [k] (cond-> {:stream stream :event k}
                                                      (= stream :user) (assoc :user-id user-id)))))))
                              set)]
    (-> predicate-events
        (conj {:stream :recommendation :event resolved})
        (conj {:stream :recommendation :event id}))))


(defn- predicate-event-ids-for-recommendation
  [resolution-status recommendation]
  (->> (completion-tests-for recommendation)
       (map #(get-in % [:test resolution-status]))
       set))

(def resolved-event-ids-for (partial predicate-event-ids-for-recommendation :resolved ))
(def unresolved-event-ids-for (partial predicate-event-ids-for-recommendation :unresolved ))

(defn resolved-recommendation-ids
  [scope]
  (->> all vals (filter #(= scope (:scope %))) (map :resolved ) set))

(defn resolved-project-recommendation-ids [] (resolved-recommendation-ids :project))
(defn resolved-user-recommendation-ids [] (resolved-recommendation-ids :user))



(comment
  (related-events-as-predicate-templates 22 confpr)
  
  )
