// @flow
import $ from 'jquery';

import {autoGrowInputList} from '../input-list';
import {validateRepoPathsOnServer} from './repo-paths';
import * as tabs from '../tabs';

function validateRepoPathsOffline() {
  const filledRepoFields = $('.repo-path-input').filter(function() { return this.value != ''; });
  const atLeastOneRepo = filledRepoFields.length >= 1;

  if (!atLeastOneRepo) {
    alert('You need to specify at least one repository path.');
  }

  return atLeastOneRepo;
}

/**
  * Checks if all repo paths conform to the git remote url syntax
  * and if their names don't conflict.
  * Check is performed _only_ if this is the project created via 'Specify Remotes'.
  */
function validateRemoteRepoPathsOnServer() {
  const localPathForRemotesInput = $('input[name="local-path-for-remotes"]'); 

  if (localPathForRemotesInput && localPathForRemotesInput.length > 0) {
    validateRepoPathsOnServer(window.location.pathname);
  }
}

function validateCodeCommentPattern() {
  var nInvalid = 0;
  $('.comment-biomarker-input').each(function () {
    const givenInput = $(this).val();

    const isEmptyString = !givenInput || /^\s*$/.test(givenInput);

    if (!isEmptyString) {
      const mustContainDescriptionAsComment = /\(\?\#[^\)]+\)$/.test(givenInput);

      if (!mustContainDescriptionAsComment) {
        ++nInvalid;
      }
    }
  });

  if (nInvalid > 0) {
    alert('All code comment patterns must be a regex with a comment describing it. That comment is used to present the results in the analysis. You have ' + nInvalid + ' expressions that lack a descriptive comment.');

    return false;
  }

  return true;
}

function validateTicketId() {
  var givenTicketPattern = $('input[name=ticketidpattern]').val();
  if (givenTicketPattern && givenTicketPattern !== '') {
    var isValidTicketPattern = /\(.+\)/.test(givenTicketPattern);

    if (!isValidTicketPattern) {
      alert('Your Ticket ID must be a regex with one (1) group to extract the matching ID');
    }

    return isValidTicketPattern;
  }

  return true;
}

function promptOverwriteOfArchDefs() {
  var givenFile =  $('input[name=importarchfile]').val();

  if (givenFile) {
    return confirm('Your existing Architectural Definitions will be overwritten with the ones from the file.');
  }

  return true;
}

function validateDynamicFormFields() {
  // notice we don't need to do server side validation of repo paths via AJAX when form is submitted
  // Any errors will be checked by the server and reported properly after the submission
  // There's no value in calling server side validation from client in this case
  return validateCodeCommentPattern() && validateRepoPathsOffline() && validateTicketId();
}


function tabLinkHandler (form: $) {
  if (HTMLFormElement.prototype.reportValidity) {
    return function (event) {
      if (! form.get(0).reportValidity()) {
        event.preventDefault();
      }
    };
  }
  else { /* Edge 16 does not support reportValidity  */
    return function (event) {
      if (! form.get(0).checkValidity()) {
        /* we only prevent the tab action here, we will report the
           error through the invalid event */
        event.preventDefault();
      }
    };
  }
}

function setInvalidEvents(form) {
  const body = $('#page-body');
  body.append('<div id="form-alerts-fixed-container"></div>');
  const container = $('#form-alerts-fixed-container');

  $(document).on('click', '#close-alerts', (e) => {
    const alertItem = $('.alert-item');
    let timeout = 0;

    alertItem.each(function (index) {
      timeout += 100 + (index * 100);
      $(this).addClass('remove-alert-item');
    });

    timeout += 100;
    $('#close-alerts').addClass('remove-alert-item');

    setTimeout(() => {
      $('.remove-alert-item').remove();
    }, timeout);
    e.preventDefault();
  });

  form.find('input').each(function (){
    this.oninvalid = (element) => {
      const closeButton = container.find('#close-alerts');
      const labelName = $(element.target).parent().find('label').text();
      const tabName = $(element.target).parents('.tab').data('tab-id');

      if (container.find(`.alert-item[data-name="${element.target.name}"]`).length) {
        return;
      }

      if (!closeButton.length) {
        container.append('<button id="close-alerts"><i class="fa fa-close"></i></button>');
      }
      container.append(
        `<div class="alert-item" data-name="${element.target.name}">
            <p class="tab-name"><span>${tabName}</span></p>
            <p><span>${labelName}</span> ${element.type}</p>
        </div>`
      );
    };
  });
}

/**
 * On form submit, an error can occur when there is invalid input in a
 * tab that is currently hidden. Chrome wants to show the invalidation
 * message and fails. Here we avoid that by validating when changing
 * tabs. This is relatively straightforward in browsers that support
 * reportValidity(). Otherwise, we cancel the navigation click and try
 * to construct an alert.
 */
function setupTabValidation() {
  const form = $('#configform');
  const tabLinks = form.find('.tab-navigation');
  tabLinks.on('click', tabLinkHandler(form));

  setInvalidEvents(form);

  if (!HTMLFormElement.prototype.reportValidity) {
    form.find('input').on('invalid', function (event: Event) {
      if (
        event.target &&
          event.target instanceof HTMLInputElement &&
          event.target.validationMessage.length > 0
      ) {
        const label = $(event.target).siblings('label');
        var labelText = 'Validation error:\n\n';
        if (label.length > 0) {
          labelText = 'Validation error in "' + label.text() + '":\n\n';
        }
        alert(labelText + event.target.validationMessage);
      }
    });
  }
}


function replaceUrl (url, preElems) {
  preElems.each(function () {
    const p = $(this);
    const txt = p.text().replace(/CS_URL/g, url);
    p.text(txt);
  });
}

function initializeStatusBadges () {
  const badgeHolder = $('.status-badge-selector, .analyzed-by-codescene');
  const matches = window.location.href.match(/^(.+)\/projects\/[0-9]+\/configuration.*$/);
  let url = '';
  if (! matches) {
    console.warn('Program error: could not extract info from location');
  }
  else {
    url = matches[1];
  }
  badgeHolder.find('.status-badge').each(
    function() {
      const badge = $(this);
      const markupContainer = badge.find('.examples-container');
      badge.find('input').change(
        function() {
          if (this.checked) {
            markupContainer.addClass('selected');
            markupContainer.removeClass('unselected');
            markupContainer.removeClass('not-selected');
          }
          else {
            markupContainer.removeClass('selected');
            markupContainer.addClass('unselected');
          }
        });

      replaceUrl(url, badge.find('pre'));
    });
}

function gitTagStrategyEnabled() {
  return $('input[id=delivery-performance-by-git-tags]:checked').val();
}

export function init(projectId : number, analysisPlan : string) {
  $('#analysisplan').cron({
    initial: analysisPlan,
    // POST expecting {'cron': '12 10 * * 6}
    url_set: `${window.baseUrl}projects/${projectId}/plan/`,
    useGentleSelect: true
  });

  autoGrowInputList($('#repository-path-list'), {
    prefix: 'vcs_',
    placeholder: 'Enter another repository path...',
    wrapperElems: [
      $('<div>', {'class': 'row repository-row'}),
      $('<div>', {'class': 'col-md-7'})
    ],
    inputFieldCss: 'repo-path-input col-md-7',
    siblings: [
      $('<div>', {'class': 'repo-path-alert col-md-7'})
    ],
    onBlur: () => validateRemoteRepoPathsOnServer()
  });

  autoGrowInputList($('#comment-biomarker-list'), {
    prefix: 'comment-biomarker-pattern-',
    placeholder: 'Enter another code comment pattern with the description as a regex comment, (?#Description of the Pattern)...',
    wrapperElems: [
      $('<div>', {'class': 'row'}),
      $('<div>', {'class': 'col-md-8'})
    ],
    inputFieldCss: 'comment-biomarker-input col-md-8'
  });

  var $couplingByTime = $('#couplingbytime');
  var $couplingByTicket = $('#couplingbyticket');

  $('input[name=ticketidpattern]').change(function() {
    var pattern = $(this).val();
    var patternIsSet = pattern && pattern !== '';

    if (!patternIsSet) {
      $couplingByTime.prop('checked', true);
      $couplingByTicket.prop('checked', false);
    }
    $couplingByTicket.prop('disabled', !patternIsSet);
  });

  $('input[type=checkbox][name=exdevcontribenabled]').change(function() {
    $('input[name=minexdevcontribtime]').prop('disabled', !this.checked);
  });

  $('input[type=checkbox][name=do_housekeeping]').change(function() {
    $('input[name=max_history_depth]').prop('disabled', !this.checked);
    $('input[name=max_historic_age]').prop('disabled', !this.checked);
  });

  $('input[type=checkbox][name=calculate-branch-statistics]').change(function() {
    $('input[name=limit_branch_prediciton_commits]').prop('disabled', !this.checked);
    $('input[name=branches-analysis-lookback-weeks]').prop('disabled', !this.checked);
    $('input[name=branch-name-exclusion-pattern]').prop('disabled', !this.checked);

    const checkLimit = this.checked && $('input[name=limit_branch_prediciton_commits]').prop('checked');
    $('input[name=max_branch_risk_history_depth]').prop('disabled', !checkLimit);  
  });

  $('input[type=checkbox][name=limit_branch_prediciton_commits]').change(function() {
    $('input[name=max_branch_risk_history_depth]').prop('disabled', !this.checked);
  });

  // Delivery performance: only enable release tags config when that option is enabled:
  $('input[name=delivery-performance-release-tag]').prop('disabled', !gitTagStrategyEnabled());

  $('input[type=radio][name=delivery-performance-strategy]').change(function() {
    $('input[name=delivery-performance-release-tag]').prop('disabled', !gitTagStrategyEnabled());
  });

  $('button.clear-temporal-coupling-filter').on('click', function () {
    const $row = $(this).parent().parent();
    $row.find('input[type=text]').val('');
    return false;
  });

  $('#configform').submit(function() {
    var progressIndicator = $('.save-progress-indicator');

    progressIndicator.removeClass('fa-check')
      .addClass('fa-circle-o-notch')
      .addClass('spinning');

    if (!validateDynamicFormFields()) {
      progressIndicator.removeClass('fa-circle-o-notch')
        .addClass('fa-exclamation-triangle')
        .removeClass('spinning');
      return false;
    }

    return promptOverwriteOfArchDefs();
  });

  const configurationTabs = $('.configuration-tabs');
  tabs.setup(configurationTabs.find('ul.nav'), configurationTabs.find('.tab-content'), 'general');
  setupTabValidation();

  // make sure that any repo path warnings (conflicting repo names) are immediatelly visible to the user.
  validateRemoteRepoPathsOnServer();
  initializeStatusBadges();
}


