(ns codescene.features.recommendations.definitions.predicates
  (:require [codescene.features.recommendations.checks :as checks]))

;;; TODO: idea: some user-facing parts of these could be defined in an
;;; external file loaded at compile time (yml maybe?)

(def predicates
  {:prs-configured?   {:id :prs-configured?
                       :doc "Are pull requests configured for an individual repository."
                       :inputs [:config]
                       :check-fn checks/prs-configured-for-repo?
                       :progress-format {:multiple "Configure PRs on %d of %d repositories"
                                         :singular "PR integration configured"
                                         :unresolved-singular "PR integration not configured"
                                         :unresolved-multiple "PR integration has not been configured for any repositories"}
                       :stream :config
                       :unresolved :no-pr
                       :resolved :pr-ok}

   :prs-received? {:id :prs-received?
                   :doc "Have any PR analyses been run yet? Maybe the PRs are set up wrong."
                   :inputs [:analysis]  ; Probably PR statistics
                   :check-fn checks/prs-received?
                   :progress-format {:singular "At least one PR has been successfully analyzed"
                                     :unresolved-singular "No PRs have been received yet"}
                   :stream :analysis
                   :unresolved :no-prs-received
                   :resolved :prs-received-ok}

;;; TODO: this should also handle the case where the file is
;;; removed. In that case, it's fine if there's no goal.
   :goal-set?   {:id :goal-set?
                 :doc "Is a goal set on the given file? Acceptable goals can be proivded via the `:goal-types` param."
                 :identifier-doc "The identifier is the complete path to the file."
                 :inputs [:goals]
                 :params {:file-path {:type :string :required true}
                          :accepted-goals {:type :goal-keyword :required false}}
                 :check-fn checks/goal-set?
                 :progress-format {:multiple "Set quality gates (goals) on %d of %d files"
                                   :singular "Set quality gate on file"
                                   :unresolved-singular "Quality gate not set"
                                   :unresolved-multiple "No quality gates set"}
                 :stream :analysis
                 :unresolved :no-goal-on-file
                 :resolved :goal-ok}

;;; TODO: this should also handle the case where the file is
;;; removed. In that case, it's fine if there's no goal.
   :no-goal-violation? {:id :no-goal-violation?
                        :doc "Has the file declined since the goal was set? This should apply to both \"Supervise\" and \"Planned refactoring\"."
                        :identifier-doc "The identifier is the complete path to the file."
                        :inputs [:goals]
                        :params {:file-path {:type :string :required true}}
                        :check-fn checks/no-goal-violation?
                        :progress-format {:multiple  "%d of %d files do not have quality gate violations"
                                          :singular  "No quality get violation detected on this file"
                                          :unresolved-singular "A goal violation was detected on this file"
                                          :unresolved-multiple "All the files have goal violations."}
                        :stream :analysis
                        :unresolved :goal-violation
                        :resolved :goal-violation-ok}

   :code-health-improved? {:id :code-health-improved?
                           :doc "Has code health improved on this goal?"
                           :identifier-doc "The identifier is a file, presumably with a goal attached."
                           :inputs [:analysis]
                           :check-fn checks/code-health-improved?
                           :stream :analysis
                           :unresolved :code-health-not-improved
                           :resolved :code-health-improved-ok}
   
;;; TODO: this one might not be a good idea. Maybe all file tracking
;;; predicates should handle the "file removed" case instead.
   :file-removed? {:id :file-removed?
                   :doc "Has the file been deleted?"
                   :identifier-doc "The identifier is the complete path to the file."
                   :inputs [:analysis]
                   :check-fn checks/file-removed?
                   :stream :analysis
                   ;; This might require some extra thought, since "not removed" is not really a problem
                   :unresolved :file-not-removed
                   :resolved :file-removed}

   :unhandled-notification-files? {:id :unhandled-notification-files?
                                   :doc "Starts with the files returned from an analysis as part of a
   given notification type (defined via a param). Checks to see that
   all of them have goals set. Primarily used for activation, not for events."
                                   :identifier-doc "List of complete file paths." ; probably not used
                                   :inputs [:goals :analysis]
                                   :params {:notification-type {:type :keyword :required true}
                                            :files {:type :list-of-strings :required true}}
                                   :check-fn checks/unhandled-notification-files?
                                   :stream :analysis
                                   :unresolved :unhandled-notification-files-found ;these probably would not be used
                                   :resolved :unhandled-notification-files-ok}

   :user-has-seen-hotspots?   {:id :user-has-seen-hotspots?
                               :doc "Has the current user seen the hotspots system map at least once?"
                               :identifier-doc "" ; TODO: not sure what to do here... is the user the identifier?
                               :inputs [:user-acts] ; TODO: user-acts is not a great name
                               :params {:user-id {:type :id :required true}}
                               :check-fn checks/user-has-seen-hotspots?
                               :progress-format {:singular  "Hotspots inspected" :unresolved-singular "Hotspots not inspected yet"}
                               :stream :user
                               :unresolved :impossible ; It doesn't make sense to have an unresolved event here.
                               :resolved :hotspots-viz-ok}})


(def event-keyword->predicate
  (into {}
        (mapcat
         (fn [{:keys [resolved unresolved] :as p}]
           (->> [resolved unresolved]
                ; remove nils
                (filter identity)
                (mapv (fn [k] [k p]))))
         (vals predicates))))

(defn unresolved-keyword-for
  "Given an event keyword (presumably the resolved one), returns the
  corresponding unresolved keyword. "
  [predicate-event-keyword]
  (:unresolved (get event-keyword->predicate predicate-event-keyword)))


(defn resolved-keyword-for
  "Given an event keyword (presumably the unresolved one), returns the
  corresponding resolved keyword. "
  [predicate-event-keyword]
  (:resolved (get event-keyword->predicate predicate-event-keyword)))

(defn resolved-kw?
  [predicate-event-keyword]
  (and predicate-event-keyword          ; avoid (= nil nil)
       (= predicate-event-keyword (:resolved (get event-keyword->predicate predicate-event-keyword)))))

(defn unresolved-kw?
  [predicate-event-keyword]
  (and predicate-event-keyword          ; avoid (= nil nil)
       (= predicate-event-keyword (:unresolved (get event-keyword->predicate predicate-event-keyword)))))



(comment
  (:unhandled-notification-files-found event-keyword->predicate)
  )

