(ns codescene.features.delta.coverage.cov-gates-db
  (:require [clojure.java.jdbc :as jdbc]
            [codescene.features.util.coll :refer [do-diff]]
            [codescene.features.util.maps :refer [map-of-db ->db-name]]
            [codescene.features.config.cov-config :as cov-config]
            [codescene.features.config.presets :as presets]
            [hugsql.core :as hugsql]))

(hugsql/def-db-fns "codescene/features/delta/coverage/cov-gates.sql")

(defn settings-id
  "Returns ID of QG settings stored at scope"
  [db-spec scope]
  (or (get-in (presets/available-settings db-spec scope ["cov-config"]) ["cov-config" 0 :id])
      (presets/create-settings {:db-spec db-spec 
                                :scope scope
                                :encryptor identity
                                :params {:category "cov-config"
                                         :label "Code Coverage Quality Gates"
                                         :qg-preset "custom"
                                         :new-and-changed-code? true}
                                :html-form-input? false})))

(defn cov-configuration
  [db-spec project-id provider-id]
  (->> (select-cov-gates db-spec (map-of-db project-id))
       (into {} (map (juxt (comp keyword :setting_key)
                           :setting_value)))
       (cov-config/from-db-cloud provider-id)))

(defn get-conf-kv
  [db-spec project-id]
  (into {} (map (juxt (comp keyword :setting_key)
                      :setting_value))
        (select-cov-gates db-spec (map-of-db project-id))))

(defn configuration-map
  "Returns full configuration for project. Scope is global settings scope.

  These properties are not deserialized!"
  [db-spec project-id props scope]
  (jdbc/with-db-transaction [tx db-spec]
    ;; settings-id function creates global settings if missing
    (let [global-id (settings-id db-spec scope)]
      (->> (presets/get-settings db-spec global-id scope)
           (cov-config/merge-configuration (not-empty (get-conf-kv tx project-id)))
           (cov-config/from-db props)))))

(defn conf->gates
  "Transforms configuration properties to a format used by CLI"
  [{:keys [coverage-metric overall-threshold changed-threshold
           overall-coverage? new-and-changed-code?] :as conf}]
  (let [->snake (comp name ->db-name)
        metric (->snake coverage-metric)]
    [{:name "overall_coverage" :threshold overall-threshold :coverage_metric metric
      :enabled overall-coverage?}
     {:name "new_and_changed_code" :threshold changed-threshold :coverage_metric metric
      :enabled new-and-changed-code?}]))

(defn update-configuration [db-spec project-id data]
  (jdbc/with-db-transaction
    [tx db-spec]
    (let [{:keys [removed added updated]} (do-diff first (get-conf-kv tx project-id) (update-vals data str))]
      (doseq [[setting-key  _] removed]
        (delete-cov-gates-prop! tx (map-of-db project-id (name setting-key))))
      (doseq [[setting-key setting-value] added]
        (insert-cov-gates-prop! tx (map-of-db project-id (name setting-key) setting-value)))
      (doseq [[_old [setting-key setting-value]] updated]
        (update-cov-gates-prop! tx (map-of-db project-id (name setting-key) setting-value))))
    project-id))