(ns codescene.features.config.pr-config
  (:require [clojure.spec.alpha :as s]
            [clojure.string :as str]
            [medley.core :as m]
            [meta-merge.core :refer [meta-merge]]
            [codescene.features.util.maps :as maps]
            [codescene.features.config.properties :refer [enum bool sval property ival] :as p]
            [codescene.features.config.sets :as sets]
            [spec-tools.data-spec :as ds]))

(s/def ::github-file-comments #{"review" "annotations" "none"})
(s/def ::qg-preset #{"global" "minimal" "medium" "all" "custom" "custom-file"})

(def properties
  {:github [(property :file-comments :github-file-comments (enum ::github-file-comments) "review")]
   :azure [(property :annotations? :azure-annotations (bool) true)
           (property :active-status? :azure-active-status (bool) true)
           (property :high-risk-active-comment? :azure-high-risk-active-comment (bool) false)]
   :bitbucket [(property :build? :bitbucket-build (bool) true)
               (property :report? :bitbucket-report (bool) true)
               (property :annotations? :bitbucket-annotations (bool) true)]
   :gitlab [(property :discussions? :gitlab-discussions (bool) true)
            (property :show-pending? :gitlab-show-pending (bool) false)
            (property :high-risk-active-comment? :gitlab-high-risk-active-comment (bool) false)]})

(def result-options [:file-comments :annotations? :active-status? :high-risk-active-comment?
                     :build? :report? :discussions? :show-pending? :always-comment? :comments?])

(def qg-properties [(property :qg-preset :qg-preset (enum ::qg-preset) "global")
                    (property :qg-hotspot-decline? :qg-hotspot-decline (bool) false)
                    (property :qg-critical-health-rules? :qg-critical-health-rules (bool) false)
                    (property :qg-advisory-health-rules? :qg-advisory-health-rules (bool) false)
                    (property :qg-new-code-health? :qg-new-code-health (bool) false)
                    (property :qg-refactoring-goals? :qg-refactoring-goals (bool) false)
                    (property :qg-supervise-goals? :qg-supervise-goals (bool) false)
                    (property :codeowners-for-critical-code? :qg-codeowners (bool) false)])

(def qg-properties-keys (mapv :id qg-properties))
(def qg-properties-keys-wo-preset (next qg-properties-keys))

(def general-delta-properties [(property :delta-ignore-other-branches :delta-ignore-other-branches (bool) false)
                               (property :delta-branch-name-exclusion-pattern :delta-branch-name-exclusion-pattern (sval))
                               (property :coupling-threshold :coupling-threshold (ival) 80)
                               (property :always-comment? :always-comment (bool) false)])

(def cloud-all-providers-properties (concat [(property :enabled? :delta-analysis-enabled (bool) false)]
                                            general-delta-properties
                                            qg-properties))

(def cloud-properties (update-vals properties #(into % cloud-all-providers-properties)))

(defn- bitbucket-server-api-url-fix [url]
  (let [slash-trimmed (str/replace-first url #"/+$" "")]
    (if (or (= url "") (str/ends-with? slash-trimmed "/rest/api/1.0"))
      slash-trimmed (str slash-trimmed "/rest/api/1.0"))))

(def onprem-properties
  (-> (m/map-keys #(get {:github "github-app" :github-hook "github"} % (name %)) properties)
      (meta-merge
        {"github-app" [(property :secret :github-secret (sval))
                       (property :private-key :github-private-key (p/file-string true))
                       (property :app-id :github-app-id (sval))
                       (property :preset-id :github-preset-id (ival))
                       (property :api-url :github-api-url (sval) "")]
         "github" [(property :password :github-deprecated-token (sval true))
                   (property :api-url :github-deprecated-api-url (sval) "")
                   (property :host-url :github-deprecated-host-url (sval))]
         "azure" [(property :password :azure-token (sval true))
                  (property :api-url :azure-api-url (sval) "")
                  (property :preset-id :azure-preset-id (ival))
                  (property :host-url :azure-host-url (sval))]
         "bitbucket" [(property :password :bitbucket-password (sval true))
                      (property :api-url :bitbucket-api-url (sval) "")
                      (property :host-url :bitbucket-host-url (sval))
                      (property :preset-id :bitbucket-preset-id (ival))
                      (property :username :bitbucket-username (sval))]
         "bitbucket-server" [(property :build? :bitbucket-server-build (bool) true)
                             (property :report? :bitbucket-server-report (bool) true)
                             (property :annotations? :bitbucket-server-annotations (bool) true)
                             (property :password :bitbucket-server-token (sval true))
                             (assoc (property :api-url :bitbucket-server-api-url (sval) "")
                               :fix-fn bitbucket-server-api-url-fix)
                             (property :preset-id :bitbucket-server-preset-id (ival))
                             (property :host-url :bitbucket-server-host-url (sval))]
         "gerrit" [(property :password :gerrit-password (sval true))
                   (property :api-url :gerrit-api-url (sval) "")
                   (property :username :gerrit-username (sval))
                   (property :comments? :gerrit-comments (bool) false)
                   (property :origin-url :gerrit-origin-url (sval))
                   (property :host-url :gerrit-host-url (sval))
                   (property :preset-id :gerrit-preset-id (ival))
                   (property :user-hook? :gerrit-user-hook (bool) false)]
         "plugin-gerrit" {}
         "gitlab" [(property :password :gitlab-token (sval true))
                   (property :api-url :gitlab-api-url (sval) "")
                   (property :host-url :gitlab-host-url (sval))
                   (property :preset-id :gitlab-preset-id (ival))]})))

(def properties-by-type
  (merge cloud-properties onprem-properties))

(defn flat-config [config] (-> config (merge (:config config)) (dissoc :config)))

(defn api-spec
  "API Spec for all configs"
  [provider-property-maps]
  (let [key-spec-coll (for [props (vals provider-property-maps)
                            {:keys [api-spec ui-param]} props]
                        [(ds/opt (maps/->db-name ui-param)) api-spec])]
    ;; the name doesn't matter, but it does show up in Swagger, so we pick a nice public one
    (ds/spec {:spec (into {} key-spec-coll) :name :pr-integration/config})))

;; Onprem pr-config has some keys at base then it has a :config key that leads to another
;; kv map
(defn ->api-representation
  "Convert webhook provider to params similar to UI"
  [all-types-map config]
  (let [config-set (sets/typed-config-set all-types-map (flat-config config) :live)]
    (:data (sets/to-ui-params config-set))))

(def onprem-base-props [:password :api-url :active? :username :type :id :project-id])

(defn onprem-split
  "Combine defaults with props and split props into onprem style base config + :config key"
  [props defaults]
  (let [m (merge defaults props)]
    (assoc (select-keys m onprem-base-props)
      :config (apply dissoc m onprem-base-props))))

(defn onprem-provider
  [props provider-type active?]
  (onprem-split props {:type provider-type :active? active? :api-url "" :username "" :password ""}))

(defn from-db-cloud
  [provider-id data]
  (p/from-db (get cloud-properties provider-id) data))

(defn from-db-onprem
  [provider-type data]
  (p/from-db (get onprem-properties provider-type) data))

