(ns codescene.features.security.tokens
  "Utilities for generating our tokens and verifying them.

  Tokens come in salted and unsalted variant"
  (:require [codescene.util.data :as data]
            [buddy.core.bytes :as bytes]
            [buddy.core.codecs :as codecs]
            [buddy.core.mac :as mac]
            [buddy.core.nonce :as nonce]
            [taoensso.timbre :as log]))

(defn- mac-options [secret]
  {:alg :hmac+sha256
   :key secret})

(defn b64-token
  "Generates a url-safe base64 encoded token without salt, using
  a value that must be convertible to bytes. Unlike salted tokens, these
  are repeatable if you can repeat the value."
  [value secret]
  (when (nil? value)
    (throw (ex-info "Unsalted HMAC token requires that you supply a value to HMAC!" {})))
  (-> (mac/hash value (mac-options secret))
      data/b64-urlsafe))

(defn b64->bytes
  [data]
  (try
    (codecs/b64->bytes data true)
    (catch Exception e
      (log/warn e "Cannot decode base64 bytes array"))))

(defn valid-b64-token?
  "Verifies a token generated by b64-token, returns true/false"
  [token value secret]
  (and (some? token)
       (some? value)
       (mac/verify value
                   (-> token codecs/str->bytes b64->bytes)
                   (mac-options secret))))

(defn b64-salted-token
  "Generates a url-safe base64 encoded token with salt, also using
  an additional value. If value is not needed, you can supply nil."
  [value secret]
  (let [salt (nonce/random-bytes 4)]
    (-> (mac/hash (bytes/concat (codecs/to-bytes value) salt)
                  (mac-options secret))
        (bytes/concat salt)
        data/b64-urlsafe)))

(defn valid-b64-salted-token?
  "Verifies a token generated by b64-salted-token, returns true/false"
  [token value secret]
  (and (some? token)
       (some? value)
       (let [token-bytes (-> token codecs/str->bytes (codecs/b64->bytes true))
             hmac (bytes/slice token-bytes 0 32)
             salt (bytes/slice token-bytes 32 36)]
         (mac/verify (bytes/concat (codecs/to-bytes value) salt)
                     hmac
                     (mac-options secret)))))

(defn b64-random-token
  "Generate a random token url-safe base64 encoded"
  []
  (-> (nonce/random-bytes 32)
      data/b64-urlsafe))

(defn b64-totp-seq
  "Generate a sequence of 32 bytes url-safe base64 encoded TOTP tokens, going backwards in time.

  Input is 32 byte secret."
  [secret]
  (letfn [(token [time-id]
            (-> (codecs/long->bytes time-id)
                (mac/hash (mac-options (cond-> secret
                                         (string? secret) (codecs/b64->bytes true))))
                data/b64-urlsafe))
          (inner [time-id]
            (cons (token time-id) (lazy-seq (inner (dec time-id)))))]
    (inner (quot (System/currentTimeMillis) 30000))))
