(ns codescene.features.reports.legacy-reports
  (:require [clojure.java.io :as io]
            [clojure.string :as str]
            [clojure.spec.alpha :as s]
            [taoensso.timbre :as log]
            [codescene.features.components.analysis :as analysis]
            [codescene.features.components.navigation :as navigation]
            [codescene.features.components.project :as project]
            [codescene.features.components.custom-report :as custom-report]
            [codescene.features.spec.commons :as commons-spec]
            [codescene.features.reports.report-types :as report-types]
            [codescene.features.reports.for_report :as for-report])
  (:import (java.util UUID)))

(s/def ::report-dir string?)
(s/def ::file-name string?)
(s/def ::full-file-name string?)
(s/def ::content-type string?)
(s/def ::name string?)

(s/def ::file-report-spec (s/keys :req-un [::report-dir ::file-name ::full-file-name ::content-type] :opt-un [::name]))
(s/def ::feature-flags (s/map-of keyword? boolean?))

(defn report-component [system]
  (:codescene.features.reports/report-component system))

;;; TODO: these should be passed in as params (in `system`?) so that
;;; we don't have to ensure that Cloud and On-prem use the same resource paths.
(def matter-sq "public/vendor/matter-sq/MatterSQ-Regular.ttf")
(def img-dir "public/imgs")

(def color-theme {:title-decoration-1 [57 133 152]
                  :title-decoration-2 [94 213 217]})

(defn- content-type
  [file-name]
  (cond
    (str/ends-with? file-name ".csv")
    {:content-type "text/csv"}

    (str/ends-with? file-name ".gz")
    {:content-type "application/x-gzip"}

    (str/ends-with? file-name ".pdf")
    {:content-type "application/pdf"}))

(defn- file-report-spec
  [report-dir file-name]
  (merge {:report-dir report-dir
          :file-name file-name
          :full-file-name (str report-dir "/" file-name)}
         (content-type file-name)))

(defn- project->report-spec
  [{:keys [analysis-component project-component navigation-component]} feature-flags project-id]
  (let [{:keys [name]} (project/project-by-id project-component project-id)
        minimal-spec {:project-id    project-id
                      :name          name
                      :feature-flags feature-flags}]
    (if-let [{:keys [id]} (analysis/latest-successful-analysis analysis-component project-id)]
      (let [path-fn (analysis/local-analysis-result-path analysis-component project-id id)
            report-dir (str "reports/" (UUID/randomUUID))
            http-base (navigation/get-http-base navigation-component)
            analysis-results-uri (navigation/get-analysis-results-uri navigation-component project-id id)
            knowledge-loss-uri (navigation/get-knowledge-loss-uri navigation-component project-id id)]
        (merge
          minimal-spec
          {:path-fn                path-fn
           :report-dir             (path-fn report-dir)
           :analysis-dashboard-url (str http-base analysis-results-uri)
           :knowledge-loss-url     (str http-base knowledge-loss-uri)
           :analysis-id            id}))
      minimal-spec)))

(defn- generate-project-report-file [report-type report-spec early-warnings architectural-components feature-flags]
  (let [{:keys [project-id analysis-id report-dir name path-fn knowledge-loss-url analysis-dashboard-url]} report-spec
        report-data {:path-fn path-fn
                     :early-warnings early-warnings
                     :architectural-components architectural-components}
        title (str (report-types/type->name report-type) " Report\nfor\n" name)
        {:keys [full-file-name] :as file-report-spec} (file-report-spec report-dir (str (str/replace title #"\n" " ") ".pdf"))
        _ (io/make-parents full-file-name)
        options {:timespan               :month
                 :title                  title
                 :font                   matter-sq
                 :colors                 color-theme
                 :img-dir                img-dir
                 :analysis-dashboard-url analysis-dashboard-url
                 :knowledge-loss-url     knowledge-loss-url}]
    (log/debugf "Generating report of type %s for project %s and analysis %s in %s."
                report-type project-id analysis-id full-file-name)
    (with-open [out (io/output-stream full-file-name)]
      (cond
        (= report-type :technical-health-overview) (for-report/for-technical-health-overview report-data options out)
        (= report-type :trend-report) (for-report/for-trend-report report-data options out)
        (and (= report-type :key-personnel-and-knowledge-distribution) (:enable-key-personnel-report feature-flags)) (for-report/for-key-personnel-and-knowledge-distribution report-data options out))
      (assoc file-report-spec :name name))))

(s/fdef generate-project-report
  :args (s/cat :duct-system map?
               :feature-flags (s/nilable ::feature-flags)
               :project-id ::commons-spec/id
               :report-type keyword?)
        :ret ::file-report-spec)
(defn generate-project-report
  [system feature-flags project-id report-type]
  (let [{:keys [analysis-component project-component navigation-component] :as component} (report-component system)
        report-spec (project->report-spec component feature-flags project-id)
        analysis-id (:analysis-id report-spec)
        http-base (navigation/get-http-base navigation-component)
        early-warnings (some->> analysis-id
                                (analysis/early-warnings analysis-component project-id)
                                (map #(assoc % :link (str http-base (navigation/get-warning-uri navigation-component project-id analysis-id (:linkable-name %))))))
        architectural-components (project/architectural-components project-component project-id)]
    (cond
      (nil? (:name report-spec))
      (throw (ex-info (format "Cannot generate the report %s, the project with id: %s
        doesn't exist or you don't have access to it" report-type project-id)
                      {:project-id project-id :report-type report-type}))

      (nil? analysis-id)
      (throw (ex-info (format "Cannot generate the report %s for project: %s(%s) when there are no analyses yet.
        Run the analysis for the project at least once." report-type (:name report-spec) project-id)
                      {:project-id project-id :report-type report-type}))

      (not (report-types/is-legacy-project-report? report-type))
      (throw (ex-info (format "Can't generate report for project id %s because the requested report type %s does not exist."
                              project-id report-type)
                      {:project-id project-id :report-type report-type}))
      :else
      (generate-project-report-file report-type report-spec early-warnings architectural-components feature-flags))))

(defn- generate-management-report-file [report-dir project-report-defs]
  (let [report-specs-by-project (into {} (map (juxt :name identity) project-report-defs))
        title "Management Overview Report"
        {:keys [full-file-name] :as file-report-spec} (file-report-spec report-dir (str title ".pdf"))
        _ (io/make-parents full-file-name)
        options {:timespan :month
                 :title    title
                 :font     matter-sq
                 :colors   color-theme
                 :img-dir  img-dir}]
    (with-open [out (io/output-stream full-file-name)]
      (for-report/for-management report-specs-by-project options out)
      file-report-spec)))

(s/fdef generate-management-report
        :args (s/cat :duct-system map? :feature-flags ::feature-flags :projects (s/coll-of (s/keys :req-un [::commons-spec/id])))
        :ret ::file-report-spec)
(defn generate-management-report [system feature-flags projects]
  (let [project-report-defs (->> projects
                                 (map #(project->report-spec (report-component system) feature-flags (:id %)))
                                 (filter :analysis-id))
        report-dir (-> (first project-report-defs) :report-dir)]
    (cond
      (empty? projects)
      (throw (ex-info "Cannot generate the management report as the requester have no accessible projects." {}))

      (empty? project-report-defs)
      (throw (ex-info "Cannot generate the management report as the accessible projects have no analysis yet. Run the analysis for your projects at least once." {}))

      :else
      (generate-management-report-file report-dir project-report-defs))))



(defn- add-with-header-to [coll]
  (map #(assoc %2 :with-header? (= 0 %1)) (iterate inc 0) coll))

(defn- write-to
  [out custom-report-component custom-report project-report-defs]
  (doseq [project-report-def (add-with-header-to project-report-defs)]
    (let [empty-buffer? (empty? (.getBuffer out))
          data (custom-report/custom-report-data
                custom-report-component
                custom-report
                (cond-> project-report-def
                        empty-buffer? (merge {:with-header? true})))]
      (.write out ^String data))))

(defn- generate-custom-report-file
  [custom-report-component custom-report project-report-defs {:keys [full-file-name] :as file-report-spec}]
  (let [_ (io/make-parents full-file-name)]
    (log/debugf "Creating report of type %s for projects in %s." (:name custom-report) full-file-name)
    (with-open [fout (io/writer full-file-name)]
      (.write fout (with-out-str (write-to *out* custom-report-component custom-report project-report-defs))))
    file-report-spec))

(s/fdef generate-custom-report
        :args (s/cat :duct-system map?
                     :projects (s/coll-of (s/keys :req-un [::commons-spec/id]))
                     :custom-report-name keyword?)
        :ret ::file-report-spec)

(defn generate-custom-report
  [system projects custom-report-name]
  (let [{:keys [custom-report-component] :as component} (report-component system)
        custom-report (custom-report/custom-report custom-report-component (name custom-report-name))
        project-report-defs (->> projects
                                 (map #(project->report-spec component {} (:id %)))
                                 (filter :analysis-id))]
    (cond
      (nil? custom-report)
      (throw (ex-info (format "The report %s does not exist." custom-report-name)
                      {:custom-report-name custom-report-name}))

      (empty? projects)
      (throw (ex-info (format "Cannot generate the %s report as the requester have no accessible projects." custom-report-name)
                      {:custom-report-name custom-report-name}))

      (empty? project-report-defs)
      (throw (ex-info (format "Cannot generate the %s report as the accessible projects have no analaysis yet. Run the analysis for your projects at least once." custom-report-name)
                      {custom-report-name custom-report-name}))

      :else
      (let [report-dir (-> (first project-report-defs)
                           :report-dir)
            multiple-projects (if (= 1 (count project-report-defs)) "" "-all-accessible-projects")
            file-report-spec (file-report-spec report-dir (str (:name custom-report) multiple-projects ".csv"))]
        (generate-custom-report-file custom-report-component custom-report project-report-defs file-report-spec)))))
