(ns codescene.features.api.routes.analyses
  (:require [compojure.api.sweet :refer [routes GET context]]
            [spec-tools.spec :as spec]
            [clojure.spec.alpha :as s]
            [codescene.features.api.privileges :as api-privileges]
            [codescene.features.api.core :as api-core]
            [codescene.features.api.spec.analysis :as api-analysis-spec]
            [codescene.features.api.analyses :as analysis]
            [clj-time.format :as f]
            [clj-time.coerce :as c]
            [clj-time.core :as t]
            [clojure.spec.alpha :as s]))

(def ^:const files-page-limit 100)
(def ^:const commits-page-limit 200)
(def ^:const issues-page-limit 200)
(def ^:const technical-debt-hotspots-page-limit 100)
(def ^:const files-default-order-by "change_frequency")

(defn- components-response-descriptor
  [text]
  {200 {:schema      ::api-analysis-spec/components_list
        :description (str "components for the " text " for the given project id")}})

(defn- component-response-descriptor
  [text]
  {200 {:schema      ::api-analysis-spec/component
        :description (str "component for the " text " for the given project id and component name")}})

(defn- component-file-list-response-descriptor
  [text]
  {200 {:schema      ::api-analysis-spec/files_list
        :description (str "components for the " text " for the given project id")}})

(defn- analysis-response-descriptor
  [text]
  {200 {:schema      ::api-analysis-spec/analysis
        :description (str text " for given project id")}})

(defn- commits-response-descriptor
  [text]
  {200 {:schema      ::api-analysis-spec/paginated-commit-list
        :description (str text " for given project id")}})

(defn- issues-response-descriptor
  [text]
  {200 {:schema      ::api-analysis-spec/paginated-issue-list
        :description (str text " for given project id")}})

(defn- files-api-response-descriptor
  [text]
  {200 {:schema      ::api-analysis-spec/files_list
        :description (str "files in decreasing order for the selected order_by criteria <br/>"
                          "for the " text " of the given project id <br/>"
                          "you can optionally set the page_size parameter, default is 100 <br/>"
                          "and the order_by to \"change_frequency\", \"code_health\", \"lines_of_code\", \"number_of_defects\" or \"cost\", <br/> "
                          "if no order by provided default is \"change_frequency\"")}})

(defn- author-statistics-descriptor []
  {200 {:schema      ::api-analysis-spec/author_statistics_response
        :description "a summary view of author contribution statistics"}})

(defn- branch-statistics-descriptor []
  {200 {:schema      ::api-analysis-spec/branch_statistics_response
        :description "a summary view of branch statistics"}})

(defn- technical-debt-hotspots-descriptor []
  {200 {:schema      ::api-analysis-spec/paginated-technical-debt-hotspots-list
        :description "a list of technical debt hotspots"}})

(defn- analysis-sub-routes
  [system project-id]
  (routes
    (GET "/" []
      :tags ["analyses"]
      :query-params [{page :- spec/pos-int? 1}]
      :responses {200 {:schema      ::api-analysis-spec/analyses_list
                       :description "analyses list for the given project id"}}
      (analysis/list-result system project-id {:page page :filter-fn (fn [x] true) :map-params {}}))
    (GET "/bydate" []
      :tags ["analyses"]
      :query-params [{page :- spec/pos-int? 1} {date :- ::api-analysis-spec/date_type ""}]
      :responses {200 {:schema      ::api-analysis-spec/analyses_list
                       :description "analyses list for the given project id filtered by given date, date format YYYY-MM-DD <br/>
                       if date parameter is missing or empty current date will be used"}}
      (let [formatter (f/formatter "yyyy-MM-dd")
            parse-timestamp-fn (fn [x] (f/unparse formatter (c/from-date x)))
            filter-date (if (s/valid? ::api-analysis-spec/date_type date) date (f/unparse formatter (t/now)))
            filter-fn (fn [x] (= filter-date (parse-timestamp-fn (get x :analysistime))))]
        (analysis/list-result system project-id {:page page :filter-fn filter-fn :map-params {:date filter-date}})))))

;; @CodeScene(disable:"Large Method")
(defn- by-id-analysis-sub-routes
  [system project-id]
  (routes
   (GET "/:analysis-id" []
     :tags ["analyses"]
     :responses (analysis-response-descriptor "analyses with id")
     :path-params [analysis-id :- spec/pos-int?]
     (analysis/analysis-result system project-id analysis-id))
   (GET "/:analysis-id/commits" []
     :tags ["analyses"]
     :responses (commits-response-descriptor "commits with id")
     :query-params [{page :- spec/pos-int? 1}
                    {page_size :- spec/pos-int? commits-page-limit}]
     :path-params [analysis-id :- spec/pos-int?]
     (analysis/commits system project-id analysis-id {:page page :page-size page_size}))
   (GET "/:analysis-id/issues" []
     :tags ["analyses"]
     :responses (issues-response-descriptor "issues with id")
     :query-params [{page :- spec/pos-int? 1}
                    {page_size :- spec/pos-int? issues-page-limit}]
     :path-params [analysis-id :- spec/pos-int?]
     (analysis/issues system project-id analysis-id {:page page :page-size page_size}))
   (GET "/:analysis-id/files" []
     :tags ["analyses"]
     :responses (files-api-response-descriptor "given analysis id")
     :path-params [analysis-id :- spec/int?]
     :query-params [{page :- spec/pos-int? 1}
                    {page_size :- spec/pos-int? files-page-limit}
                    {order_by :- ::api-analysis-spec/order_by files-default-order-by}
                    {fields :- spec/string? ""}
                    {filter :- spec/string? ""}]
     (analysis/analysis-files-list-result
      system project-id
      analysis-id
      {:page page
       :page-size page_size
       :order-by order_by
       :fields fields
       :filter filter}))
   (GET "/:analysis-id/commit-activity" []
     :summary "Fetch a list of the weekly commit trends"
     :tags ["analyses"]
     :responses {200 {:schema      ::api-analysis-spec/commit_activity_trend_response
                      :description "a weekly trend over the revisions and contribution authors"}}
     :path-params [analysis-id :- spec/int?]
     (-> (analysis/get-analysis system project-id analysis-id)
         analysis/commit-activity-trend))
   (GET "/:analysis-id/author-statistics" []
     :summary "Fetch a list of author statistics"
     :tags ["analyses"]
     :responses (author-statistics-descriptor)
     :path-params [analysis-id :- spec/int?]
     (analysis/author-statistics system project-id analysis-id))
   (GET "/:analysis-id/branch-statistics" []
     :summary "Fetch a list of branch statistics"
     :tags ["analyses"]
     :responses (branch-statistics-descriptor)
     :path-params [analysis-id :- spec/int?]
     (analysis/branch-statistics system project-id analysis-id))
   (GET "/:analysis-id/technical-debt-hotspots" []
     :summary "Fetch technical debt hotspots"
     :tags ["analyses"]
     :responses (technical-debt-hotspots-descriptor)
     :query-params [{page :- spec/pos-int? 1}
                    {page_size :- spec/pos-int? technical-debt-hotspots-page-limit}
                    {order_by :- ::api-analysis-spec/order_by files-default-order-by}
                    {filter :- spec/string? ""}]
     :path-params [analysis-id :- spec/int?]
     (analysis/technical-debt-hotspots
      system
      project-id
      analysis-id
      {:page page
       :page-size page_size
       :order-by order_by
       :filter filter}))))

;; @CodeScene(disable:"Large Method")
(defn- latest-analysis-sub-routes
  [system project-id]
  (routes
   (GET "/latest" []
     :tags ["analyses"]
     :responses (analysis-response-descriptor "latest analysis")
     (analysis/latest-analysis-result system project-id))
   (GET "/latest/files" []
     :tags ["analyses"]
     :responses (files-api-response-descriptor "latest analysis")
     :query-params [{page :- spec/pos-int? 1}
                    {page_size :- spec/pos-int? files-page-limit}
                    {order_by :- ::api-analysis-spec/order_by files-default-order-by}
                    {fields :- spec/string? ""}
                    {filter :- spec/string? ""}]
     (analysis/latest-analysis-files-list-result 
      system 
      project-id 
      {:page page 
       :page-size page_size 
       :order-by order_by
       :fields fields
       :filter filter}))
   (GET "/latest/commit-activity" []
     :summary "Fetch a list of the weekly commit trends"
     :tags ["analyses"]
     :responses {200 {:schema      ::api-analysis-spec/commit_activity_trend_response
                      :description "a weekly trend over the revisions and contribution authors"}}
     (analysis/latest-commit-activity-trend system project-id))
   (GET "/latest/commits" []
     :summary "Fetch a list of commits"
     :tags ["analyses"]
     :responses (commits-response-descriptor "commits with id")
     :query-params [{page :- spec/pos-int? 1}
                    {page_size :- spec/pos-int? commits-page-limit}]
     (analysis/latest-analysis-commits system project-id page page_size))
   (GET "/latest/issues" []
     :summary "Fetch a list of issues"
     :tags ["analyses"]
     :responses (issues-response-descriptor "issues with id")
     :query-params [{page :- spec/pos-int? 1}
                    {page_size :- spec/pos-int? issues-page-limit}]
     (analysis/latest-analysis-issues system project-id page page_size))
   (GET "/latest/author-statistics" []
     :summary "Fetch a list of author statistics"
     :tags ["analyses"]
     :responses (author-statistics-descriptor)
     (analysis/latest-author-statistics system project-id))
   (GET "/latest/branch-statistics" []
     :summary "Fetch a list of branch statistics"
     :tags ["analyses"]
     :responses (branch-statistics-descriptor)
     (analysis/latest-branch-statistics system project-id))
   (GET "/latest/technical-debt-hotspots" []
     :summary "Fetch technical debt hotspots"
     :tags ["analyses"]
     :responses (technical-debt-hotspots-descriptor)
     :query-params [{page :- spec/pos-int? 1}
                    {page_size :- spec/pos-int? technical-debt-hotspots-page-limit}
                    {order_by :- ::api-analysis-spec/order_by files-default-order-by}
                    {filter :- spec/string? ""}]
     (analysis/latest-technical-debt-hotspots 
      system 
      project-id 
      {:page page 
       :page-size page_size
       :order-by order_by
       :filter filter}))))

(defn- component-latest-analysis-sub-routes
  [system project-id]
  (routes
    (GET "/latest/components" []
      :tags ["analyses"]
      :query-params [{page :- spec/pos-int? 1}]
      :responses (components-response-descriptor "latest analysis")
      (analysis/latest-components-list-result system project-id page))
    (GET "/latest/components/:component" []
      :tags ["analyses"]
      :responses (component-response-descriptor "latest analysis")
      :path-params [component :- spec/string?]
      (analysis/latest-component-result system project-id component))
    (GET "/latest/components/:component/files" []
      :tags ["analyses"]
      :responses (component-file-list-response-descriptor "latest analysis")
      :path-params [component :- spec/string?]
      :query-params [{page :- spec/pos-int? 1}
                     {page_size :- spec/pos-int? files-page-limit}
                     {order_by :- ::api-analysis-spec/order_by files-default-order-by}
                     {fields :- spec/string? ""}
                     {filter :- spec/string? ""}]
      (analysis/latest-components-file-list-result
       system
       project-id
       component
       {:page page
        :page-size page_size
        :order-by order_by
        :fields fields
        :filter filter}))))

(defn- component-by-id-analysis-sub-routes
  [system project-id]
  (routes
    (GET "/:analysis-id/components" []
      :tags ["analyses"]
      :query-params [{page :- spec/pos-int? 1}]
      :responses (components-response-descriptor "analysis with id")
      :path-params [analysis-id :- spec/int?]
      (analysis/components-list-result system project-id analysis-id page))
    (GET "/:analysis-id/components/:component" []
      :tags ["analyses"]
      :responses (component-response-descriptor "analysis with id")
      :path-params [analysis-id :- spec/int? component :- spec/string?]
      (analysis/component-result system project-id analysis-id component))
    (GET "/:analysis-id/components/:component/files" []
      :tags ["analyses"]
      :responses (component-file-list-response-descriptor "analysis with id")
      :path-params [analysis-id :- spec/int? component :- spec/string?]
      :query-params [{page :- spec/pos-int? 1}
                     {page_size :- spec/pos-int? files-page-limit}
                     {order_by :- ::api-analysis-spec/order_by files-default-order-by}
                     {fields :- spec/string? ""}
                     {filter :- spec/string? ""}]
      (-> (analysis/get-analysis system project-id analysis-id)
          (analysis/components-file-list-result 
           component 
           {:page page 
            :page-size page_size 
            :order-by order_by 
            :fields fields
            :filter filter})))))

(defn sub-routes
  [system project-id]
  (context "/analyses" []
    :middleware [#(api-core/wrap-authorize-project system % project-id #{api-privileges/restapi-read})
                 analysis/wrap-csv-not-found]
    (analysis-sub-routes system project-id)
    (latest-analysis-sub-routes system project-id)
    (component-latest-analysis-sub-routes system project-id)
    (by-id-analysis-sub-routes system project-id)
    (component-by-id-analysis-sub-routes system project-id)))
