// @flow
import $ from 'jquery';

type OnChangeFunc  = (element : $) => void;
type OnBlurFunc  = (element : $) => void;
type OnGrowFunc  = (element : $) => void;

export type AutoGrowInputListOptions = {
  prefix? : string,
  placeholder? : string,
  wrapperElems? : Array<any>,
  // onChange function will be executed with every change of input field including onblur event
  onChange? : OnChangeFunc,
  // onBlur function is executed _only_ with onblur event
  onBlur? : OnBlurFunc,
  inputFieldCss? : string,
  // list of immediate siblings of input field elements - e.g. div for showing validation message
  siblings? : Array<any>,
  onGrow? : OnGrowFunc
};

export function autoGrowInputList(list : (Element | $ | string), options : AutoGrowInputListOptions) {
  options = options || {};

  var prefix = options.prefix || '';
  var placeholder = options.placeholder || '';
  var inputFieldCss = options.inputFieldCss || '';

  // wrapperElems should be a list of jQuery elements that will wrap
  // the newly created input field. The first element wraps the
  // second, etc. The inner-most element will be the input field.
  var wrapperElems = options.wrapperElems || null;

  var siblings = options.siblings || null;
  var $list = $(list);

  function onInputChange() {
    growListWhenNeeded();
    if (typeof options.onChange === 'function') {
      const onChange : OnChangeFunc = options.onChange;
      onChange($(this));
    }
  }

  function onInputBlur() {
    if (typeof options.onBlur === 'function') {
      const onBlur : OnBlurFunc = options.onBlur;
      onBlur($(this));
    }
  }

  function wrapInput (wrapperElems, $e, siblings) {
    var $elem = $e;
    for (var i = wrapperElems.length - 1; i > -1; i--) {
      var $wrap = wrapperElems[i].clone();
      $elem = $wrap.append($elem);
      if (i === wrapperElems.length - 1 && siblings) {
        // add siblings as children of the innermost wrapper element
        siblings.forEach((s) => $wrap.append(s.clone()));
      }
    } 
    return $elem;
  }

  $list.find('input').on('change blur keyup', onInputChange);
  $list.find('input').on('blur', onInputBlur);

  function growListWhenNeeded() {
    var $fields = $list.find('input[name^="' + prefix + '"]');

    var emptyFieldCount = $fields.filter(function() { return this.value == ''; }).length;

    if (emptyFieldCount < 1) {
      var $emptyInputField = $('<input>')
        .attr('class', 'form-control ' + inputFieldCss)
        .attr('type', 'text')
        .attr('name', prefix + $fields.length)
        .attr('placeholder', placeholder)
        .attr('value', '')
        .on('change blur keyup', onInputChange)
        .on('blur', onInputBlur);
      if (wrapperElems && wrapperElems.length > 0) {
        $list.append(wrapInput(wrapperElems, $emptyInputField, siblings));
      } else {
        $list.append($emptyInputField);
        if (siblings) {
          siblings.forEach((s) => $list.append(s.clone()));
        }
      }
      if (typeof options.onGrow === 'function') {
        const onGrow : OnGrowFunc = options.onGrow;
        onGrow($(this));
      }
    }
  }
  $list.find('input').keyup(growListWhenNeeded);
  growListWhenNeeded();
}
