grammar CStyleNestedComplexity;

// This grammar detects islands of nested complexity. It's run on a function scope.


// Pre-processor directives and conditional compilation:
// ====================================================
//
// This is incredibly complicated. Basically, we need to parse all compilation paths. Otherwise we might
// miss out on nested complexity that's located under an #ifdef.
//
// The main challenge is that pre-proc directives are allowed to break conditional blocks
// (see https://trello.com/c/Dtl6HiqF/1442-parse-nested-logic-that-crosses-pre-processor-boundaries-in-c).
//
// This means we need to parse the same function multiple times, once for each compilation path.
// In this grammar, we define different entry points and the application logic is responsible for
// invoking them and accumulating the results. The main impact is on DEEP NESTED LOGIC and BUMPY ROAD.

// Here are the entry points for the different application paths:
//
pre_proc_if_method: pre_proc_if_block_expression* EOF;
pre_proc_else_method: pre_proc_else_block_expression* EOF;


// Grammar for pre-proc directives, option 1: #if path
//

pre_proc_if_block_expression: skip_pre_proc_else_block
                            | pre_proc_if_complexity
                            | block_expression_if_pre_proc_context
                            | pre_proc_if_block_anything;
pre_proc_if_block_anything: ~(OPENING_BRACE | CLOSING_BRACE | PRE_PROC_ELSE | PRE_PROC_ELIF);

pre_proc_if_complexity: pre_proc_if_if_clause
                      | pre_proc_if_else_clause
                      | pre_proc_if_switch_clause
                      | pre_proc_if_for_range
                      | pre_proc_if_for_loop
                      | pre_proc_if_for_range_with_initializer
                      | pre_proc_if_while_loop
                      | pre_proc_if_do_while_loop;

pre_proc_if_if_clause: IF some_condition (pre_proc_if_multi_line_conditional_expression | plain_line);
pre_proc_if_else_clause: ELSE (pre_proc_if_multi_line_conditional_expression | plain_line);

pre_proc_if_switch_clause: SWITCH some_condition pre_proc_if_multi_line_conditional_expression;

pre_proc_if_multi_line_conditional_expression: OPENING_BRACE pre_proc_if_block_expression*? CLOSING_BRACE;

pre_proc_if_for_range: FOR for_range_part ':' conditions+? RIGHT_PAREN
           (pre_proc_if_multi_line_conditional_expression | plain_line);

// Range-based for loop with initializer
//  for (std::vector v{1, 2, 3}; auto& e : v) {
pre_proc_if_for_range_with_initializer: FOR LEFT_PAREN for_range_with_initializer_data_part ';'
                                        for_range_with_initializer_variable ':' conditions+? RIGHT_PAREN
                                       (pre_proc_if_multi_line_conditional_expression | plain_line);

pre_proc_if_for_loop: FOR LEFT_PAREN for_loop_part SEMICOLON for_loop_part SEMICOLON
                       for_loop_condition RIGHT_PAREN (pre_proc_if_multi_line_conditional_expression | plain_line);

pre_proc_if_while_loop: WHILE some_condition (pre_proc_if_multi_line_conditional_expression | plain_line);
pre_proc_if_do_while_loop: DO pre_proc_if_multi_line_conditional_expression WHILE some_condition;

block_expression_if_pre_proc_context: OPENING_BRACE pre_proc_if_block_expression*? CLOSING_BRACE;

skip_pre_proc_else_block: (PRE_PROC_ELSE | PRE_PROC_ELIF) pre_proc_body*? pre_proc_endif;

// Option 2: the #else of #elif compilation paths
//
// Grammar for pre-proc directives, option 1: #if path
//

pre_proc_else_block_expression: skip_pre_proc_if_block
                              | pre_proc_else_complexity
                              | block_expression_else_pre_proc_context
                              | pre_proc_else_block_anything;
pre_proc_else_block_anything: ~(OPENING_BRACE | CLOSING_BRACE | PRE_PROC_IF | PRE_PROC_IFDEF | PRE_PROC_IFNDEF);

pre_proc_else_complexity: pre_proc_else_if_clause
                        | pre_proc_else_else_clause
                        | pre_proc_else_switch_clause
                        | pre_proc_else_for_range
                        | pre_proc_else_for_loop
                        | pre_proc_else_for_range_with_initializer
                        | pre_proc_else_while_loop
                        | pre_proc_else_do_while_loop;

pre_proc_else_if_clause: IF some_condition (pre_proc_else_multi_line_conditional_expression | plain_line);
pre_proc_else_else_clause: ELSE (pre_proc_else_multi_line_conditional_expression | plain_line);

pre_proc_else_switch_clause: SWITCH some_condition pre_proc_else_multi_line_conditional_expression;

pre_proc_else_multi_line_conditional_expression: OPENING_BRACE pre_proc_else_block_expression*? CLOSING_BRACE;

pre_proc_else_for_range: FOR for_range_part ':' conditions+? RIGHT_PAREN // ~(OPENING_BRACE)+? RIGHT_PAREN?
           (pre_proc_else_multi_line_conditional_expression | plain_line);

// Range-based for loop with initializer
//  for (std::vector v{1, 2, 3}; auto& e : v) {
pre_proc_else_for_range_with_initializer: FOR LEFT_PAREN for_range_with_initializer_data_part ';'
                                        for_range_with_initializer_variable ':' conditions+? RIGHT_PAREN
                                       (pre_proc_else_multi_line_conditional_expression | plain_line);

pre_proc_else_for_loop: FOR LEFT_PAREN for_loop_part SEMICOLON for_loop_part SEMICOLON
                       for_loop_condition RIGHT_PAREN (pre_proc_else_multi_line_conditional_expression | plain_line);

pre_proc_else_while_loop: WHILE some_condition (pre_proc_else_multi_line_conditional_expression | plain_line);
pre_proc_else_do_while_loop: DO pre_proc_else_multi_line_conditional_expression WHILE some_condition;

block_expression_else_pre_proc_context: OPENING_BRACE pre_proc_else_block_expression*? CLOSING_BRACE;

skip_pre_proc_if_block: (PRE_PROC_IF | PRE_PROC_IFDEF | PRE_PROC_IFNDEF)
                         pre_proc_body*?
                         (PRE_PROC_ELSE | PRE_PROC_ELIF | PRE_PROC_ENDIF);


// Shared parser rules.
//

pre_proc_endif: PRE_PROC_ENDIF;

pre_proc_body : pre_proc_block
              | any_statement;
any_statement : ~(PRE_PROC_IF | PRE_PROC_IFDEF | PRE_PROC_IFNDEF);
pre_proc_block : (PRE_PROC_IF | PRE_PROC_IFDEF | PRE_PROC_IFNDEF) pre_proc_condition? pre_proc_block_statement*? (PRE_PROC_ELSE | PRE_PROC_ENDIF);
pre_proc_block_statement : (PRE_PROC_IF | PRE_PROC_IFDEF | PRE_PROC_IFNDEF) pre_proc_condition? pre_proc_block_statement*? PRE_PROC_ENDIF
                         | ~(PRE_PROC_ENDIF | PRE_PROC_IF | PRE_PROC_IFDEF | PRE_PROC_IFNDEF);
pre_proc_condition : ('!' | '1' | '0' | ID);

// Shared rules for nested complexity.
//

plain_line: ~OPENING_BRACE ~(SEMICOLON)*? SEMICOLON;

for_range_part: LEFT_PAREN ~(':' | ';' | OPENING_BRACE)+?;

for_range_with_initializer_data_part: ~(';')+?; // Parse things like std::vector v{1, 2, 3}
for_range_with_initializer_variable: ~(':' | ';' | OPENING_BRACE)+?;

for_loop_part: ~(SEMICOLON)*?; // each part is optional in C
for_loop_condition: conditions*?;

some_condition: LEFT_PAREN conditions+ RIGHT_PAREN;
conditions : conditional_operator
           | ~(LEFT_PAREN | RIGHT_PAREN | SEMICOLON | OPENING_BRACE | IF | ELSE | FOR | WHILE | DO | SWITCH)
           | LEFT_PAREN conditions*? RIGHT_PAREN;

conditional_operator: CONDITIONAL_OPERATORS;

fragment ESCAPED : '\\\\' | '\\"';
LITERAL : '"' ( ESCAPED | ~('\n'|'\r') )*? '"';

SHARP : '#';

PRE_PROC_IF : SHARP 'if';
PRE_PROC_IFDEF : SHARP 'ifdef';
PRE_PROC_IFNDEF : SHARP 'ifndef';

PRE_PROC_ELSE : SHARP 'else';
PRE_PROC_ELIF:  SHARP 'elif';

PRE_PROC_ENDIF: SHARP 'endif';

IF: 'if';
ELSE: 'else';
FOR: 'for';
WHILE: 'while';
DO: 'do';
SWITCH: 'switch';

CONDITIONAL_OPERATORS: '&&' | '||';

ID : [a-zA-Z_][a-zA-Z0-9_]*;

LEFT_PAREN: '(';
RIGHT_PAREN: ')';
OPENING_BRACE: '{';
CLOSING_BRACE: '}';
COLON: ':';
SEMICOLON: ';';

Whitespace : [ \t]+ -> skip;

BlockComment: '/*' .*? '*/' -> skip;
LineComment: '//' ~[\r\n]* -> skip;

NEWLINE : '\r'? '\n' -> skip;

ANY_CHAR : .; // Put this lexer rule last to give it the lowest precedence