(ns codescene.features.tools.licensing
  (:require [buddy.sign.jwt :as buddy-jwt]
            [codescene.features.util.jwt :as jwt]
            [clj-time.core :as tc]
            [clj-time.coerce :as tcc]
            [integrant.core :as ig]
            [medley.core :as m]))

(defprotocol LicenseProvider
  (-cli-license [this user])
  (-vs-code-license [this user]))

(defn claims [duration leeway issuer aud]
  (m/assoc-some {:iss issuer
                 :iat (tcc/to-epoch (tc/now))}
    :exp (some->> duration (tc/plus (tc/now)) tcc/to-epoch)
    :aud aud
    ;; leeway is how many seconds after expiry the validation still works
    :leeway (when leeway (tc/in-seconds leeway))))

(defn jwt [claims keyring] (jwt/make-jwt claims nil keyring))

(defn expired? [claims] (<= (:exp claims) (tcc/to-epoch (tc/now))))

(defn verified-claims [license license-keyring]
  (let [typ (-> (jwt/extract-jwt license) :header :typ)]
    (if typ
      (jwt/unpack-jwt license license-keyring)
      ;; older onprem versions emit no typ
      (buddy-jwt/unsign license
                        (some #(when (= "csent" (:kid %)) (:public-key %)) license-keyring)
                        {:alg :eddsa}))))

(defn- user-claims [user]
  (m/assoc-some {}
    :sub (:nickname user)
    :acc (:account-id user)
    :usr (:user-id user)
    :identity (:identity user)))

(defrecord JWTLicenseProvider [f]
  LicenseProvider
  (-cli-license [this user] (f user "codescene-cli"))
  (-vs-code-license [this user] (f user "codescene-vscode")))

(defn jwt-license-provider [issuer keyring duration-fn]
  (let [keyring (jwt/init-keyring keyring)]
    (->JWTLicenseProvider
      (fn [user product]
        (jwt (merge (claims (duration-fn user) nil issuer product) (user-claims user))
             keyring)))))

(defn cli-license [provider user] (-cli-license provider user))
(defn vs-code-license [provider user] (-vs-code-license provider user))

(defmethod ig/init-key ::no-licensing [_ _v])

(derive ::no-licensing :codescene.features.tools.components/licensing)