(ns codescene.features.repository-provider.github.checks
  "Operations to deal with the GitHub Run Checks API: developer.github.com/v3/checks/runs.

  This API requires GitHub App for authentication - legacy OAuth apps aren't supported!
  That means we can't leverage our usual 'CodeScene Repositories' application
  but we need to use/authorize a separate app.
  See `codescene.features.repository-provider.github.app`.

  Note on Retries: This module doesn't deal with retries since they may not be needed.
  (checks isn't typically a critical API and it's better to stay simple instead of blocking threads).
  Therefore we don't use Hystrix but plain clj-http."
  (:require [codescene.features.client.api :refer [to-url]]
            [codescene.features.repository-provider.github.api :as api]
            [medley.core :refer [assoc-some]]
            [taoensso.timbre :as log]))

(defn trim-body [^String text]
  ;; There's a 64kb limit by GitHub
  (if (< 64000 (alength (.getBytes text)))
    (let [arr (byte-array 64000)]
      (System/arraycopy (.getBytes text) 0 arr 0 64000)
      (str (String. ^bytes arr) "... OMITTED DUE TO SIZE"))
    text))

(defn create-check-run
  "Given API url for a github repo like `https://api.github.com/repos/jumarko/poptavka`
  creates a new check_run with given name and associated with given commit SHA.
  `:details-url` can be used to provide a specific link to the check run results on our website.
  Returns a complete response body.
  See https://developer.github.com/v3/checks/runs/#create-a-check-run."
  [installation-token-auth repo-url {:keys [name head-commit-sha external-id details-url] :as check-params}]
  (api/api-request installation-token-auth
                   :post
                   (to-url repo-url "check-runs")
                   ;; TODO ROK This is just for onprem users with GitHub enterprise server that is old. This preview has graduated
                   {:accept "application/vnd.github.antiope-preview+json"
                    :form-params (assoc-some {:name name
                                              :external_id external-id
                                              :head_sha head-commit-sha}
                                             :details_url details-url)}))

(defn update-check-run
  "Updates the state of given check-run as defined by `update-params`.
  Returns a complete response body.
  See https://developer.github.com/v3/checks/runs/#update-a-check-run."
  [installation-token-auth check-run-url update-params]
  (log/debugf "Updating checkrun %s" check-run-url)
  (doseq [annotation-pack (or (->> update-params :output :annotations (partition-all 50) seq) [[]])]
    (api/api-request installation-token-auth
                     :patch
                     check-run-url
                     ;; conclusion will automatically set the 'status' field to 'completed'
                     ;; output: https://developer.github.com/v3/checks/runs/#output-object-1
                     ;; actions: https://developer.github.com/v3/checks/runs/#actions-object
                     ;; - actions could be useful if we had "auto-fix" functionality in CodeScene;
                     ;;   but perhaps even without that we can suggest looking at hotspots, virtual code review, etc.
                     ;; TODO ROK This is just for onprem users with GitHub enterprise server that is old. This preview has graduated
                     {:form-params (cond-> update-params
                                     (:output update-params) (assoc-in [:output :annotations] annotation-pack))
                      :accept "application/vnd.github.machine-man-preview+json"})))

(defn get-check-run
  "Returns check run data.
  See https://developer.github.com/v3/checks/runs/#update-a-check-run."
  [installation-token-auth check-run-url]
  (api/api-request installation-token-auth :get check-run-url {}))

(defn rerequest-check-run
  "Trigger check run rerequest
  See https://docs.github.com/en/rest/checks/runs#rerequest-a-check-run."
  [installation-token-auth check-run-url]
  (api/api-request installation-token-auth
                   :post
                   (to-url check-run-url "rerequest")
                   ;; conclusion will automatically set the 'status' field to 'completed'
                   ;; output: https://developer.github.com/v3/checks/runs/#output-object-1
                   ;; actions: https://developer.github.com/v3/checks/runs/#actions-object
                   ;; - actions could be useful if we had "auto-fix" functionality in CodeScene;
                   ;;   but perhaps even without that we can suggest looking at hotspots, virtual code review, etc.
                   {}))
