(ns codescene.features.reports.graphics.chart
  (:require [clj-pdf.utils :refer [get-color]]
            [tick.core :as t]
            [codescene.features.reports.common.style-def :as style-def]
            [codescene.util.time :as ut])
  (:import (java.awt.geom Ellipse2D$Double)
           (java.text DateFormat DecimalFormat NumberFormat SimpleDateFormat)
           [org.jfree.chart ChartFactory]
           (org.jfree.chart.axis DateAxis DateTickUnit DateTickUnitType)
           (org.jfree.chart.labels StandardXYItemLabelGenerator)
           [org.jfree.chart.plot DatasetRenderingOrder IntervalMarker XYPlot]
           (org.jfree.chart.ui Layer)
           (org.jfree.data.time TimeSeriesCollection TimeSeries Day)
           (org.jfree.chart.renderer.xy XYLineAndShapeRenderer)
           [java.awt.geom Ellipse2D$Double]
           [java.awt BasicStroke Font]
           [java.time.temporal ChronoUnit]
           [java.util Date]))



(defn- set-background-look [chart [lower upper] beginning end]
  (let [^XYPlot plot (.getPlot chart)
        y-axis (.getRangeAxis plot)
        ^DateAxis x-axis (.getDomainAxis plot)
        font (Font. Font/SANS_SERIF Font/PLAIN 5)
        risky-marker (IntervalMarker. (- lower 1000) lower (get-color style-def/pale-risky-color) (BasicStroke.) nil nil 1.0)
        attention-marker (IntervalMarker. lower upper (get-color style-def/pale-attention-color) (BasicStroke.) nil nil 1.0)
        healthy-marker (IntervalMarker. upper (+ upper 1000) (get-color style-def/pale-healthy-color) (BasicStroke.) nil nil 1.0)]
    (.setTickLabelFont x-axis font)
    (.setTickUnit x-axis (DateTickUnit. DateTickUnitType/DAY 1))
    (.setDateFormatOverride x-axis (SimpleDateFormat. "d MMM"))
    (.setVerticalTickLabels x-axis true)
    (.setRange x-axis ^Date beginning ^Date end )
    (.setLowerMargin x-axis 5)
    (.setUpperMargin x-axis 5)
    (.setVisible y-axis false)
    (.setLowerMargin y-axis 0.5)
    (.setUpperMargin y-axis 0.5)
    (.setOutlinePaint plot (get-color style-def/light-grey))
    (.addRangeMarker plot risky-marker (Layer/BACKGROUND))
    (.addRangeMarker plot attention-marker (Layer/BACKGROUND))
    (.addRangeMarker plot healthy-marker (Layer/BACKGROUND))))


(defn- set-dot-series-look [renderer color]
  (let [dot-shape (Ellipse2D$Double. -2.5 -2.5 5.0 5.0)
        font (Font. Font/SANS_SERIF Font/BOLD 7)
        ^DateFormat date-format (DateFormat/getInstance)
        ^NumberFormat number-format (DecimalFormat. "#0.000")]
    (.setUseOutlinePaint renderer true)
    (.setSeriesOutlinePaint renderer 0 (get-color style-def/medium-grey))
    (.setSeriesPaint renderer 0 (get-color color))
    (.setSeriesShape renderer 0 dot-shape)
    (.setSeriesItemLabelGenerator renderer 0 (StandardXYItemLabelGenerator. "{2}" date-format number-format))
    (.setSeriesItemLabelFont renderer 0 font)
    (.setSeriesItemLabelPaint renderer 0 (get-color color))
    (.setSeriesItemLabelsVisible renderer 0 true)))


(defn- set-series-look [chart]
  (let [^XYPlot plot (.getPlot chart)
        trend-data-renderer (XYLineAndShapeRenderer. true false)
        risky-renderer (XYLineAndShapeRenderer. false true)
        attention-renderer (XYLineAndShapeRenderer. false true)
        healthy-renderer (XYLineAndShapeRenderer. false true)]
    (.setRenderer plot 0 trend-data-renderer)
    (.setSeriesPaint trend-data-renderer 0 (get-color style-def/medium-grey))
    (.setSeriesShapesVisible trend-data-renderer 0 false)
    (.setDatasetRenderingOrder plot (DatasetRenderingOrder/FORWARD))
    (.setRenderer plot 1 risky-renderer)
    (set-dot-series-look risky-renderer style-def/risky-color)
    (.setRenderer plot 2 attention-renderer)
    (set-dot-series-look attention-renderer style-def/attention-color)
    (.setRenderer plot 3 healthy-renderer)
    (set-dot-series-look healthy-renderer style-def/healthy-color)))


(defn- create-chart [all-data risky-data attention-data healthy-data]
  (let [chart (ChartFactory/createTimeSeriesChart "" "" "" all-data false false false)
        ^XYPlot plot (.getPlot chart)]
    (.setDataset plot 1 risky-data)
    (.setDataset plot 2 attention-data)
    (.setDataset plot 3 healthy-data)
    chart))


(defn- trend-item->time-series-data-item [{:keys [value date]}]
  (let [instant (ut/instant date)
        day-value (t/day-of-month instant)
        month (t/month instant)
        month-value (.getValue month)
        year (t/year instant)
        year-value (.getValue year)
        day (Day. ^int day-value ^int month-value ^int year-value)]
    [day value]))


(defn- trend->time-series [trend name]
  (let [time-series-data-items (map trend-item->time-series-data-item trend)
        time-series (TimeSeries. name)]
    (doseq [[day value] time-series-data-items]
      (.addOrUpdate time-series ^Day day ^double value))
    (TimeSeriesCollection. time-series)))


(defn code-health-trend [{:keys [line-data risky-data attention-data healthy-data]} score-bounds
                         beginning end]
  (let [all-dataset (trend->time-series line-data :all-data)
        risky-dataset (trend->time-series risky-data :risky-data)
        attention-dataset (trend->time-series attention-data :attention-data)
        healthy-dataset (trend->time-series healthy-data :healthy-data)
        chart (create-chart all-dataset risky-dataset attention-dataset healthy-dataset)
        beginning-date (Date/from (.minus beginning 1 (ChronoUnit/DAYS)))
        end-date (Date/from (.plus end 1 (ChronoUnit/DAYS)))]
    (set-background-look chart score-bounds beginning-date end-date)
    (set-series-look chart)
    chart))

