(ns codescene.features.reports.custom-chart
  "Chart types that weren't available in clj-pdf.charting using the :chart tag is implemented in this namespace.
  Ideally, this should be moved to inside clj-pdf, made available using tags, and submitted as a merge request"
  (:require [clj-pdf.utils :refer [get-color]]
            [clj-pdf.graphics-2d :as g2d])
  (:import [org.jfree.chart ChartFactory ChartUtilities JFreeChart]
           [org.jfree.chart.plot XYPlot PlotOrientation CategoryPlot DatasetRenderingOrder]
           [org.jfree.data.xy XYSeries XYSeriesCollection]
           [org.jfree.chart.renderer.category BarRenderer LineAndShapeRenderer]
           org.jfree.data.category.DefaultCategoryDataset
           org.jfree.data.general.DefaultPieDataset
           org.jfree.chart.renderer.category.StandardBarPainter
           org.jfree.chart.labels.StandardXYItemLabelGenerator
           [org.jfree.chart.axis DateAxis NumberAxis NumberTickUnit]
           java.text.SimpleDateFormat
           java.text.NumberFormat
           java.io.ByteArrayOutputStream
           java.awt.Rectangle
           (org.jfree.data.time TimeSeriesCollection TimeSeries Day Week)
           (org.jfree.chart.renderer.xy XYLineAndShapeRenderer XYBarRenderer)))

(defn- set-background [^JFreeChart chart color]
  (when-let [color (get-color color)]
    (-> chart .getPlot (.setBackgroundPaint color))))

(defn- date-within-limits [date date-start date-end]
  (or (not date-start) (not date-end)
      (and (< 0 (.compareTo date date-start))
           (< 0 (.compareTo date-end date)))))

(defn- data->Weeks [formatter [title & points]]
  (let [series (new TimeSeries title)]
    (doseq [[x y] points]
      (.addOrUpdate series
                    (new Week (.parse formatter x))
                    (double y)))
    (new TimeSeriesCollection series)))

(defn brooks-law-chart [{:keys [^String title
                           background
                           time-format
                           x-label
                           y-label
                           y2-label]} & data]
  (let [^java.text.DateFormat formatter (new SimpleDateFormat
                                             (or time-format "yyyy-MM-dd-HH:mm:ss"))
        datasets (map #(data->Weeks formatter %) data)
        ^DateAxis domain-axis (DateAxis. x-label)
        ^NumberAxis range-axis (NumberAxis. y-label)
        ^NumberAxis range2-axis (when y2-label (NumberAxis. y2-label))
        barRenderer (new XYBarRenderer 0.2)
        lineRenderer (new XYLineAndShapeRenderer true false)
        plot (new XYPlot (first datasets) domain-axis range-axis barRenderer)
        chart (new JFreeChart title JFreeChart/DEFAULT_TITLE_FONT plot true)]

    ;; Apply default theme before customizing appearance...
    (.apply (ChartFactory/getChartTheme) chart)

    ;; Plot properties
    (when-let [color (get-color background)]
      (.setBackgroundPaint plot color))
    (.setDatasetRenderingOrder plot DatasetRenderingOrder/FORWARD)

    ;; Renderer properties
    (.setShadowVisible barRenderer false)

    ;; First dataset properties
    (.setAutoRangeIncludesZero range-axis false)

    ;; Add second dataset and set properties
    (.setDataset plot 1 (second datasets))
    (.setRenderer plot 1 lineRenderer)
    (.setRangeAxis plot 1 range2-axis)
    (.mapDatasetToRangeAxis plot 1 1)
    (.setAutoRangeIncludesZero range2-axis false)

    ;; Set ticks and intervals
    (.setStandardTickUnits range-axis (NumberAxis/createIntegerTickUnits))
    (.setStandardTickUnits range2-axis (NumberAxis/createIntegerTickUnits))

    (.setVerticalTickLabels domain-axis true)

    (comment (-> plot
                 ^DateAxis (. getDomainAxis)
                 ^void (. setDateFormatOverride formatter)))

    chart))