(ns codescene.features.pm-data.azure.azure-api
  "Contains methods for fetching data over the Azure Rest API.
   Queries and paging but nothing more is handled here, with the requested json data returned as clojure collections"
  (:require
   [clojure.string :as str]
   [codescene.features.util.http-helpers :as h]
   [codescene.features.client.api :refer [check-interrupt]]
   [codescene.features.repository-provider.azure.api :as azure-api]
   [evolutionary-metrics.trends.dates :as dates]
   [medley.core :as m]
   [taoensso.timbre :as log]))

(defn delete
  ([api-client url]
   (delete api-client url {}))
  ([api-client url query-params]
   (h/with-http-error-messages (str "Failed to delete Azure data from " url)
     (azure-api/api-request api-client :delete url
                            {:query-params query-params}))))
(defn get-data
  ([api-client url]
   (get-data api-client url {}))
  ([api-client url query-params]
   (h/with-http-error-messages (str "Failed to fetch Azure data from " url)
     (azure-api/api-request api-client :get url
                            {:query-params query-params}))))

(defn post-data
  ([api-client url form-params]
   (post-data api-client url form-params {}))
  ([api-client url form-params query-params]
   (h/with-http-error-messages (str "Failed to fetch Azure data from " url)
     (azure-api/api-request api-client :post url
                            {:query-params query-params
                             :form-params form-params}))))

(defn- get-paged-data [api-client url query-params]
  (loop [next-url url
         previous-data []]
    (check-interrupt)
    (log/infof "Fetching paged azure data from %s..." next-url)
    (let [{:keys [isLastBatch nextLink values]} (get-data api-client next-url query-params)
          all-data (concat previous-data values)]
      (if-not isLastBatch
        (recur nextLink all-data)
        all-data))))

(defn- get-paged-data-using-skip 
  [api-client url query-params top]
  (loop [previous-data []]
    (check-interrupt)
    (log/infof "Fetching paged azure data from %s, starting on %s..." url (count previous-data))
    (let [query-params (m/assoc-some query-params
                                     :$skip (count previous-data)
                                     :$top top)
          {:keys [value]} (get-data api-client url query-params)
          all-data (into previous-data value)]
      (if (and (seq value)
               (or (nil? top) (> top (count all-data))))
        (recur all-data)
        all-data))))

(defn- post-paged-data 
  ([api-client url form-params]
   (post-paged-data api-client url form-params {}))
  ([api-client url form-params query-params]
   (loop [next-url url
          previous-data []]
     (check-interrupt)
     (log/infof "Fetching paged azure data from %s, got %s items so far..." next-url (count previous-data))
     (let [{:keys [isLastBatch nextLink values]} (post-data api-client next-url form-params query-params)
           all-data (concat previous-data values)]
       (if-not isLastBatch
         (recur nextLink all-data)
         all-data)))))

(defn fetch-projects
  "Fetches projects from the remote Azure API. Throws when the API calls fail."
  [api-client owner]
  (let [url (format "/%s/_apis/projects" owner)]
    (->> (get-data api-client url)
         :value)))

(defn fetch-fields
  "Fetches fields from the remote Azure API. Throws when the API calls fail."
  [api-client owner]
  (let [url (format "/%s/_apis/wit/fields" owner)]
    (->> (get-data api-client url)
         :value)))

(defn fetch-boards
  "Fetches boards from the remote Azure API. Throws when the API calls fail."
  [api-client owner project]
  (let [url (format "/%s/%s/_apis/work/boards" owner project)]
    (->> (get-data api-client url)
         :value)))

(defn fetch-board-columns
  "Fetches board columnns from the remote Azure API. Throws when the API calls fail."
  [api-client owner]
  (let [url (format "/%s/_apis/work/boardcolumns" owner)]
    (->> (get-data api-client url)
         :value)))

(defn fetch-work-item-types
  "Fetches board columnnsfrom the remote Azure API. Throws when the API calls fail."
  [api-client owner project]
  (let [url (format "/%s/%s/_apis/wit/workitemtypes" owner project)]
    (->> (get-data api-client url)
         :value)))

(defn query-work-items
  "Queries for work items from the remote Azure API. Throws when the API calls fail."
  [api-client owner project work-item-types]
  (let [url (format "/%s/%s/_apis/wit/wiql" owner project)
        form-params {:query (format "SELECT [Id] FROM WorkItems WHERE [Work Item Type] IN (%s)"
                                    (->> work-item-types
                                         (map #(format "'%s'" %))
                                         (str/join ",")))}
        query-params {:api-version "2.0"}]
    (->> (post-data api-client url form-params query-params)
         :workItems)))

(defn fetch-work-items
  "Fetches work items from the remote Azure API. Throws when the API calls fail."
  [api-client owner project item-ids]
  (let [url (format "/%s/%s/_apis/wit/workitems" owner project)
        query-params {:ids (str/join "," item-ids)}]
    (->> (get-data api-client url query-params)
         :value)))

(defn fetch-work-item-revisions
  "Fetches board columnnsfrom the remote Azure API. Throws when the API calls fail."
  [api-client owner project item-id]
  (let [url (format "/%s/%s/_apis/wit/workitems/%d/revisions" owner project item-id)]
    (->> (get-data api-client url)
         :value)))

(defn- not-empty-coll [coll]
  (not (and (coll? coll) (empty? coll))))

(defn fetch-reporting-work-item-revisions
  "Fetches reporting work item revisions from the remote Azure API. Throws when the API calls fail."
  [api-client search-options result-options]
  (let [{:keys [owner project work-item-types since]} search-options
        {:keys [fields]} result-options
        url (format "/%s/%s/_apis/wit/reporting/workitemrevisions" owner project)
        form-params (m/assoc-some {:includeLatestOnly true}
                                  :fields (not-empty fields)
                                  :types (not-empty work-item-types))
        query-params (merge {:api-version "2.0"}
                            (when since {:startDateTime (dates/date-time->string since)}))]
    (if (not-empty-coll work-item-types)
      (post-paged-data api-client url form-params query-params)
      [])))

(defn fetch-reporting-work-item-links
  "Fetches reporting work item links from the remote Azure API. Throws when the API calls fail."
  [api-client search-options]
  (let [{:keys [owner project link-types since]} search-options
        url (format "/%s/%s/_apis/wit/reporting/workitemlinks" owner project)
        query-params (merge {:linkTypes link-types}
                            (when since {:startDateTime (dates/date-time->string since)}))]
    (get-paged-data api-client url query-params)))

(defn fetch-repositories
  "Fetches repositories from the remote Azure API. Throws when the API calls fail."
  [api-client owner project]
  (let [url (format "/%s /%s/_apis/git/repositories" owner project)
        query-params {}]
    (->> (get-data api-client url query-params)
         :value)))

(defn fetch-pull-requests
  "Fetches completed pull requests from the remote Azure API. Throws when the API calls fail."
  [api-client {:keys [owner project repo]} {:keys [status count] :as _search-options}]
  (let [url (format "/%s/%s/_apis/git/repositories/%s/pullrequests" owner project repo)
        query-params (if status {:searchCriteria.status status} {})]
    (get-paged-data-using-skip api-client url query-params count)))

(defn fetch-pull-request
  "Fetches pull request from the remote Azure API. Throws when the API calls fail."
  [api-client {:keys [owner project repo]} pr]
  (let [url (format "/%s/%s/_apis/git/repositories/%s/pullrequests/%s" owner project repo pr)
        query-params {:includeCommits "true"}]
    (get-data api-client url query-params)))

(comment
  (def api-client ["" (System/getenv "AZURE_TOKEN")])
  (def owner "empear")
  (def project "SmartHotel360");650a76c2-d1f4-4085-a6b8-d87bf4e5a6bd
  (def project "dcd6233e-c108-4b8a-bb00-5f39a1a978f0") ; project with spaces
  (def repo "PublicWeb");cacf0ab2-c089-4ba1-9095-7e0fda291027
  (def repo "Repo With Spaces")

  (def api-client nil)
  (def owner "github")
  (def project "cpython")
  (->> (fetch-pull-requests api-client {:owner owner :project project :repo repo} 
                            {:status "completed"
                             :count 1})
       (map :pullRequestId))
  (->> (fetch-pull-request api-client {:owner owner :project project :repo repo} 9)
       :completionOptions :mergeCommitMessage)
  (fetch-work-item-revisions api-client owner project 1)
  (->> (fetch-reporting-work-item-revisions api-client
                                            {:owner owner
                                             :project project
                                             :work-item-types ["Task"]
                                             :since (dates/date-time-string->date "2020-02-25T13:40:00Z")}
                                            {:fields ["System.CreatedDate" "System.ChangedDate" "System.State" "Microsoft.VSTS.Scheduling.Effort"]})
       (map :fields)
       (map :System.ChangedDate))
  (fetch-reporting-work-item-links api-client {:owner owner
                                               :project project
                                               :link-types ["System.LinkTypes.Hierarchy"]
                                               :since (dates/date-time-string->date "2020-02-25T13:40:00Z")})
  (fetch-work-items api-client owner project (range 1 10))
  (query-work-items api-client owner project ["Task"])
  (fetch-work-item-types api-client owner project)
  (fetch-board-columns api-client owner)
  (fetch-boards api-client owner project)
  (fetch-fields api-client owner)
  (fetch-projects api-client owner)
  (fetch-repositories api-client owner project)
  )
