(ns codescene.features.plugins.virtual-code-reviewer-extension
  "Wraps an external VirtualCodeReviewerExtension in an implementation of 
  the VirtualCodeReviewerExtension protocol defined in codescene-analysis."
  (:require [codescene.features.plugins.configuration :as c]
            [codescene.features.plugins.system-map-extension :refer [->File ->Repository]]
            [codescene.extensions.virtual-code-reviewer-extension :refer [VirtualCodeReviewerExtension]]
            [clj-codescene-plugin.virtual-code-reviewer :as virtual-code-reviewer]
            [hotspots-x-ray.recommendations.code-health.descriptions.shared-vocabulary :as vocabulary]
            [clojure.string :as str]
            [clojure.set :refer [rename-keys]]
            [medley.core :as m]
            [taoensso.timbre :as log]))

(defn- ->hotspot-data [hotspot-name]
  (when-let [i (str/index-of hotspot-name "/")]
    {:repo (subs hotspot-name 0 i)
     :path (subs hotspot-name (inc i))}))

(defn- ->Context
  "Provides a Context instance for the external SystemMapExtension"
  [{:keys [settings encryptor] :as _context}]
  (virtual-code-reviewer/->Context {:settings (c/decrypt-values encryptor settings)}))

(def ->analysis-indication {:good vocabulary/indication-good
                            :warning vocabulary/indication-warning
                            :problem vocabulary/indication-problem})

(defn description->markdown
  [description]
  (str/join "\n\n" description))

(defn fix->markdown
  [fix]
  (->> fix
       (map (fn [{:keys [description link]}]
              (str description
                   (when link
                     (format " [Read More](%s)" link)))))
       (str/join "\n\n")))

(defn- Recommendation->map 
  "Handle naming differences between codescene-plugin naming and analysis naming"
  [r]
  (-> (virtual-code-reviewer/Recommendation->map r)
      (update :indication ->analysis-indication)
      (rename-keys {:why :description
                    :how-to-fix :fix})
      (m/update-existing :description description->markdown)
      (m/update-existing :fix fix->markdown)))

(defn- recommendations [plugin-id analyzer hotspot]
  (try
    (let [hotspot-data (->hotspot-data hotspot)]
      (->> (.getRecommendations analyzer (->Repository hotspot-data) (->File hotspot-data))
           (map Recommendation->map)))
    (catch Exception e
      (let [msg (format "Failed to get recommendations in plugin %s" plugin-id)]
        (log/warnf e msg)
        []))))

(defn ->VirtualCodeReviewerExtension [plugin-id extension context]
  (try
    (let [analyzer (.getAnalyzer extension (->Context context))]
      (reify VirtualCodeReviewerExtension
        (-id [this] plugin-id)
        (-recommendations [this hotspot] (recommendations plugin-id analyzer hotspot))))
    (catch Exception e
      (let [msg (format "Failed to create extension in plugin %s" plugin-id)]
        (log/warnf e msg)))))

(comment
  (def hotspot "analysis-target/a/c-example.c" 
    #_{:module "analysis-target/a/c-example.c"
       :revisions "2"
       :code "5"
       :id "analysis-target/c-example.c"})
  (def context {:settings []})
  (def remedy (reify com.codescene.plugin.virtualcodereviewer.Remedy
                        (getDescription [_] "some-remedy")
                        (getLink [_] "http://somewhere.com")))
  (def recommendation (reify com.codescene.plugin.virtualcodereviewer.Recommendation
                (getType [_] com.codescene.plugin.virtualcodereviewer.RecommendationType/PROBLEM)
                (getWhere [_] "somewhere")
                (getTitle [_] "Some Title")
                (getWhy [_] ["Some reason"])
                (getHowToFix [_] [remedy])))
  (def ext (reify com.codescene.plugin.VirtualCodeReviewerExtensionPoint
             (getAnalyzer [_ _] 
               (reify com.codescene.plugin.virtualcodereviewer.Analyzer
                 (getRecommendations [_ _repo _file]
                   [recommendation])))))
  (def e (->VirtualCodeReviewerExtension "dummy-plugin" ext context))
  (codescene.extensions.virtual-code-reviewer-extension/recommendations e hotspot))