(ns codescene.features.delta.result.pull-request
  "Code for delta result interpretation for the purposes of PR integrations"
  (:require [clj-time.core :as t]
            [clj-time.format :as tf]
            [clojure.spec.alpha :as s]
            [clojure.string :as str]
            [codescene.delta.delta-result :as result]
            [codescene.delta.detectors.quality-gates :as qg]
            [codescene.delta.specs :as delta-specs]
            [codescene.features.delta.quality-gates :as quality-gates]
            [codescene.features.delta.suppress :as suppress]
            [codescene.features.util.template :as template]
            [codescene.features.util.maps :refer [map-of]]
            [codescene.features.util.url :as url-utils]
            [codescene.specs :as base-specs]
            [evolutionary-metrics.mining.file-patterns :as file-patterns]
            [evolutionary-metrics.analysis.math :as math]
            [hotspots-x-ray.recommendations.code-health.rule-weights :as rule-weights]
            [medley.core :as m]
            [selmer.util :refer [without-escaping]]
            [taoensso.timbre :as log]))

(s/def ::result-url (s/nilable ::base-specs/uri))
(s/def ::result-redirect-url (s/nilable ::base-specs/uri))
(s/def ::delta-scope (s/keys :req-un [::result-url ::result-redirect-url]
                             :opt-un [::suppress/suppressions]))
(s/def ::result-options (s/keys :opt-un [::always-comment? ::include-review-findings?]))

(defn negative-review? [pr-presentable]
  (if (->> pr-presentable :result :file-results (some :enabled-gates))
    (-> pr-presentable :qg-summary :fail not-empty)
    (-> pr-presentable :summary :negative-review?)))

(defn result-keep-keys [_options]
  [:version
   :external-review-id
   :absent-changes
   :high-risk?
   :delta-branch-head
   :code-health-alert-level
   :code-owners-for-quality-gates
   ;; we don't need this but keep for logging purposes and the filtered-findings
   ;; closes over this anyway
   :file-results
   ;; due to rest api
   :code-health-delta-descriptions
   :id :warnings :description :improvements :new-files-info
   :quality-gates :qg-meta :risk])

(defn pp-impact-description [{pp :new-pp :as _finding}]
  (let [pp (or pp 0)]
    (cond
      (< 4 pp) "High"
      (< 2 pp) "Medium"
      :else "Low")))


(def default-icon-set {:h2-pass "✅" :pass "✅" :fail "❌" :h2-fail "❌"})

(defn title-prefix-for
  [{:keys [change-level change-type]}]
  (str (case change-level
         :warning \u274c
         :notice \u2139
         :improvement \u2705
         nil) " "
       (case change-type
         "introduced" "New issue:"
         "degraded" "Getting worse:"
         "improved" "Getting better:"
         "fixed" "No longer an issue:")))

(defn title-for
  [{:keys [category]} cd]
  (str (title-prefix-for cd) " " category))

(defn finding-list-item
  "A keep-findings predicate that creates an item for the purposes of findings list"
  [file-result {:keys [category]} {:keys [change-level new-pp locations] :as cd}]
  {:category category
   :hotspot? (:hotspot? file-result)
   :new-pp new-pp
   :change-level change-level
   :file-name (:name file-result)
   :short-file-name (file-patterns/final-segment (:name file-result))
   :function (quality-gates/finding-function locations)
   :impact-desc (str (pp-impact-description cd) " impact")})

(defn category-description [{:keys [change-type category] :as _finding}]
  ;; when an issue is completely fixed, don't say it's a problem
  (when (not= change-type "fixed")
    (rule-weights/description category)))

(defn file-comment-item
  "A keep-findings predicate that creates a context for generating  for the purposes of findings list"
  [{:keys [finding-url-fn include-code-health-impact? _icons]} locator]
  (fn [file-result {:keys [category] :as finding} {:keys [change-level description new-pp locations] :as cd}]
    (let [file-name (:name file-result)
          location (locator file-result cd)
          finding-url (finding-url-fn category file-name (quality-gates/finding-function locations))]
      (when (not= :improvement change-level)
        {:category category
         :description description
         :include-code-health-impact? include-code-health-impact?
         :impact-desc (str (pp-impact-description cd) " impact")
         :title-prefix (title-prefix-for cd)
         :change-level change-level
         :file-name file-name
         :new-pp new-pp
         :old-file-name (:old-name file-result)
         :location location
         :function (or (:function location) (some :function (:locations finding)))
         :finding-url finding-url
         :suppress-url (when (= :warning change-level)
                         (str finding-url "&suppress=true"))}))))

(defn file-comment-items
  "Returns file comment contexts"
  [{:keys [filtered-findings] :as pr-presentable} locator]
  (filtered-findings (file-comment-item pr-presentable locator)))

(defn findings-list
  "Create up to two lists of findings for display in the main comment with display texts."
  [findings]
  (let [findings (result/sort-findings findings)
        warnings (filter #(= :warning (:change-level %)) findings)
        improvements (filter #(= :improvement (:change-level %)) findings)]
    (cond-> []
      (seq warnings) (conj {:title "\uD83D\uDEA9 Declining Code Health (highest to lowest)"
                            :items warnings})
      (seq improvements) (conj {:title "✅ Improving Code Health"
                                :items improvements}))))

(defn cached-locator
  "Returns a fn that, given a file-results and change-details will return the location of the finding."
  [diff->locator]
  (let [cache (volatile! {})]
    (fn [file-result cd]
      (if-let [locator (@cache (:name file-result))]
        (locator cd)
        (let [l (diff->locator (:diff file-result))]
          (vreset! cache (assoc @cache (:name file-result) l))
          (l cd))))))

(defn summary [{:keys [code-health-alert-level]} filtered-findings]
  (let [findings (filtered-findings (fn [{:keys [hotspot?] :as file-result} _ {:keys [change-level]}]
                                      {:change-level change-level
                                       :hotspot-filename (when hotspot? (:name file-result))}))
        {:keys [warning improvement notice] :or {warning 0 improvement 0 notice 0}}
        (frequencies (map :change-level findings))]
    {:issues-detected warning
     :issues-improved improvement
     :has-findings? (pos-int? (+ warning notice improvement))
     :negative-review? (boolean (first (filtered-findings (result/negative-review-pred code-health-alert-level))))
     :affected-hotspots (->> (keep :hotspot-filename findings)
                             distinct
                             count)}))

(s/def ::issues-detected nat-int?)
(s/def ::issues-improved nat-int?)
(s/def ::negative-review? boolean?)
(s/def ::has-findings? boolean?)
(s/def ::result ::delta-specs/delta-analysis-result)
(s/def ::result-url (s/nilable ::base-specs/uri))
(s/def ::summary (s/keys :req-un [::has-findings? ::affected-hotspots ::issues-detected ::issues-improved
                                  ::negative-review?]))

(s/def ::delta-result-presentable (s/keys :req-un [::result ::result-url ::summary]))

(defn map-directives
  [directives]
  (map (fn [{:keys [rules fn-name]}]
         {:rules (str/join "; " rules)
          :fn-name fn-name})
       directives))

(defn directives [file-results]
  (reduce (fn [acc {:keys [directives name]}] 
            (cond-> acc
              (seq (:added directives)) (update :added conj {:name name
                                                             :directives (map-directives (:added directives))})
              (seq (:removed directives)) (update :removed conj {:name name
                                                                 :directives (map-directives (:removed directives))})))
          {:added [] :removed []}
          file-results))

(defn with-finding-list
  [pr-presentable]
  (update pr-presentable :filtered-findings #(findings-list (% finding-list-item))))

(defn failed-gate-markdown [{:keys [gate causes] :as qg} finding-url-fn provider-url-fn qg-meta]
  {:gate gate
   :gate-name (qg/gate-names gate)
   :desc (quality-gates/description qg (:new-file-threshold qg-meta))
   :causes (->> causes
                (sort-by quality-gates/score-change)
                (mapv #(quality-gates/cause-printed % gate finding-url-fn provider-url-fn)))})

(defn improvement-printed [{:keys [review name old-name] :as file-result}]
  {:file-name (file-patterns/final-segment (or name old-name))
   :code-health-impact (quality-gates/score-change-str file-result)
   :categories (str/join ", " (map :category review))})

(defn improvements-markdown [presentable]
  (let [improvements (->> (-> presentable :result :file-results)
                          (result/xf-results (fn [{:keys [enabled-gates]} finding cd]
                                               ;; qg related for improvements are improvements on files
                                               ;; with any gates turned on
                                               (when (and (result/qg-related? enabled-gates finding cd)
                                                          (= (result/change-level-for finding cd) :improvement))
                                                 :improvement))))
        files (into []
                    (comp (filter result/improved?) ;; CS-5275
                          (map improvement-printed))
                    improvements)]
    {:files-improved (count files)
     :files files}))

(defn silence-advisory-critical-pairs
  "Prevent advisory rule violations being reported if also subject of critical rule reporting"
  [failed-gates]
  (let [critical-rule-idxs (if-let [gate (m/find-first #(= "critical-health-rules" (:gate %)) failed-gates)]
                             (set (result/change-details false (fn [_ _ {:keys [idx]}] idx) (:causes gate)))
                             #{})
        silence-cd (fn [_ _ {:keys [idx]}] (nil? (critical-rule-idxs idx)))]
    (->> failed-gates
         (mapv (fn [qg]
                 (if (= (:gate qg) "advisory-health-rules")
                   (update qg :causes (partial result/xf-results silence-cd))
                   qg)))
         (remove (comp empty? :causes))
         vec)))

(defn qg-summary->markdown-parts [{:keys [finding-url-fn qg-meta] :as presentable} provider-url-fn]
  (let [add-markdown #(failed-gate-markdown % finding-url-fn provider-url-fn qg-meta)]
    (-> presentable
        (update-in [:qg-summary :fail] silence-advisory-critical-pairs)
        (update-in [:qg-summary :fail] (partial mapv add-markdown))
        (assoc :improvements (improvements-markdown presentable)))))

(s/fdef transform-results
        :args (s/cat :result ::delta-specs/delta-analysis-result
                     :delta-scope ::delta-scope
                     :options ::result-options)
        :ret map?)

(def public-icon-set
  (let [url #(format "[![](https://codescene.io/imgs/svg/%s)](#)" %)]
    {:h2-pass (url "pass.svg") :pass (url "pass1.svg") :fail (url "x1.svg") :h2-fail (url "x.svg")}))

(defn svg-icons? [external-review-url]
  (#{"github.com" "gitlab.com" "dev.azure.com"}
   (:host (url-utils/url->parts external-review-url))))

(defn mk-finding-url-fn [result-redirect-url]
  (fn [biomarker filename method]
    (some-> result-redirect-url
            (url-utils/add-params (map-of biomarker filename method)))))

(defn transform-results
  "Prepares results for output, based on options.

  Calculates and groups findings under :findings key, removes all large collections from result,
  calculates some general flags in summary."
  [{:keys [commits file-results qg-meta] :as delta-analysis-result}
   {:keys [result-url result-redirect-url suppressions qg-config-url external-review-url]}
   options]
  (let [filtered-findings #(result/keep-findings (result/unsuppressed? %) suppressions file-results)
        directives (directives file-results)
        scores (result/file-scores delta-analysis-result)
        result (-> (select-keys delta-analysis-result (result-keep-keys options))
                   (update :delta-branch-head
                           (fn [head commits]
                             ;; use delta-branch-sha if it's one of the commits otherwise use last commit
                             (or (m/find-first #(= head %) commits) (last commits)))
                           commits))
        codescene-url (url-utils/base-url result-url)]
    {:result result
     :code-owners-for-cc (quality-gates/cc-codeowners-to-notify file-results)
     :code-health (result/weighted-score scores :score :loc)
     :old-code-health (result/weighted-score scores :old-score :old-loc)
     :filtered-findings filtered-findings
     :directives directives
     :qg-meta (assoc qg-meta :qg-profile-name (qg/gate-profile-names (:qg-profile qg-meta))
                             ;;
                             :empty-results? (empty? file-results))
     :qg-config-url (str codescene-url qg-config-url)
     :qg-summary (quality-gates/qg-summary suppressions file-results)
     :summary (summary result filtered-findings)
     :icons (if (svg-icons? external-review-url)
              public-icon-set
              default-icon-set)
     :result-url result-url
     :doc-cli-page (url-utils/make-url codescene-url "/docs/cli/index.html")
     :doc-pr-page (url-utils/make-url codescene-url "/docs/guides/technical/code-health.html#adapt-code-health-to-your-coding-standards")
     :finding-url-fn (mk-finding-url-fn result-redirect-url)}))

(def default-options
  {:include-review-findings? true})

(s/fdef comment?
        :args (s/cat :result ::delta-result-presentable
                     :options (s/? ::result-options))
        :ret boolean?)
(defn comment?
  "Encapsulates the rules for posting result for a PR analysis."
  [{{:keys [issues-detected]} :summary {:keys [absent-changes]} :result :as pr-presentable}
   {:keys [include-review-findings? always-comment?] :as _options}]
  (boolean (or always-comment? (negative-review? pr-presentable) (seq absent-changes)
               ;; Only comment if there are actionable findings
               (and include-review-findings? (pos-int? issues-detected)))))

(defn render-error-title [error]
  (let [ex-obj (-> error ex-data)]
    (case (:type ex-obj)
      :http-error
      (if (#{401 403} (:status ex-obj))
        "Invalid or expired project owner's credentials."
        "Internal error.")

      :delta-analysis/result-not-found
      "No full analysis done yet."

      :delta-analysis/account-blocked
      "Account blocked."

      :delta-analysis/account-trial-expired
      "Account trial has expired."

      :delta-analysis/repo-not-found
      "Could not find repository"

      :delta-analysis/multiple-enabled-projects
      "Conflicting projects with Pull Request integration"

      :delta-analysis/branch-excluded
      "Branch is excluded from PR Integration by the configured filter."

      :permission-denied
      "Permission denied"

      :git-branch-head-moved
      "Commit is no longer head of the branch."

      :max-commits-limit
      "Max commits exceeded."

      :max-files-limit
      "Max modified files exceeded."

      "Internal error.")))

(def check-failed-message "The check failed due to an internal error. Retry the check later or go to help center: https://helpcenter.codescene.com/")

(defn render-error [error]
  (let [ex-obj (-> error ex-data)]
    (case (:type ex-obj)
      :http-error
      (if (#{401 403} (:status ex-obj))
        (format "Could not fetch commits due to an invalid access token or insufficient access. If you're using CodeScene Cloud, the account owner needs to visit https://codescene.io/users/me/token to refresh their token. Git provider reported: %s \n\n" (ex-message error))
        (format "The check failed due to a HTTP error at %s, with message %s. Retry the check later or go to help center: https://helpcenter.codescene.com/, time: %s"
                (:url ex-obj)
                (ex-message error)
                (tf/unparse (tf/formatters :date-time) (t/now))))

      (:delta-analysis/result-not-found :delta-analysis/account-blocked :delta-analysis/account-trial-expired
                                        :delta-analysis/repo-not-found :delta-analysis/multiple-enabled-projects :delta-analysis/branch-excluded
                                        :git-branch-head-moved :max-commits-limit :max-files-limit)

      (ex-message error)

      :permission-denied
      (:friendly-message error)

      (str check-failed-message
           ", time: " (tf/unparse (tf/formatters :date-time) (t/now))))))

(s/fdef presentable->comment
  :args (s/cat :result ::delta-result-presentable
               :provider-id keyword?
               :description-details? boolean?)
  :ret string?)
(defn presentable->comment
  [pr-presentable provider-id description-details?]
  ;; bitbucket has a bug where line items that start bolded don't draw a bullet for
  ;; lines after the first so "* **some item**" x3 will draw one bullet unless you double
  ;; newline, but that produces extra spacing in GitLab and such, so we limit it to Bitbucket
  (template/render-file "templates/delta/comment.md"
                        (assoc (qg-summary->markdown-parts (with-finding-list pr-presentable) (constantly nil))
                          :bottom-hr (case provider-id
                                       :azure "<hr>"
                                       :bitbucket "\n"
                                       "##      ")
                          :collapsible (if (#{:bitbucket :gerrit} provider-id) false true)
                          :extra-newline (if (#{:bitbucket :gitlab :azure} provider-id) "\n" "")
                          :description-details? description-details?)))

(s/fdef presentable->text-comment
  :args (s/cat :result ::delta-result-presentable
               :description-details? boolean?)
  :ret string?)
(defn presentable->text-comment
  [pr-presentable description-details?]
  (let [context (assoc (qg-summary->markdown-parts (with-finding-list pr-presentable) (constantly nil))
                  :description-details? description-details?)
        rendered (without-escaping (template/render-file "templates/delta/comment.txt" context))]
    ;; stupid selmer cannot remove empty rows left by excluded if tags or for tags
    (str/replace rendered #"\n{2,}" "\n\n")))
