(ns codescene.features.code-coverage.check.check-db
  (:require [clojure.edn :as edn]
            [clojure.spec.alpha :as s]
            [codescene.features.api.spec.common :as common-spec]
            [codescene.features.util.maps :refer [map-of-db ->db-keys ->kebab]]
            [codescene.specs :as specs]
            [hugsql.core :as hugsql]
            [clojure.java.jdbc :as jdbc]
            [taoensso.timbre :as log])
  (:import (java.sql SQLException)))

(s/def ::repo-id (specs/string-n 2048))
(s/def ::commit-sha ::specs/commit-hash)
(s/def ::base-ref (s/nilable (specs/string-n 512)))
(s/def ::coverage-results any?)
(s/def ::results (s/keys :req-un [::repo-id ::commit-sha ::base-ref ::coverage-results]))
(s/def ::date (s/and string? #(not (empty? %)) #(re-matches common-spec/YYYY-MM-DD %)))

(hugsql/def-db-fns "codescene/features/code_coverage/check/check.sql")

(defn result-coord [results] (select-keys results [:repo-id :commit-sha :base-ref :project-id]))

(defn get-by-props
  "Returns a check by using properties (repo-id, commit, ref)."
  [tx project-id {:keys [repo-id commit-sha base-ref]}]
  (some-> (get-check tx (map-of-db project-id repo-id commit-sha base-ref))
          ->kebab
          (update :metadata edn/read-string)))

(defn update-with-results
  "Updates check result."
  [db-spec project-id results]
  (let [data (-> results
                 (assoc :project-id project-id)
                 (update :coverage-results #(some-> % pr-str))
                 (dissoc :metadata)
                 (->db-keys))]
    (jdbc/with-db-transaction [tx db-spec]
      (if (:base-ref results)
        (do (log/infof "Updating Coverage Check results project-id=%s, results=%s" project-id (result-coord results))
            (update-results tx data))
        (try (log/infof "Inserting Coverage Run results project-id=%s, results=%s" project-id (result-coord results))
             (insert-results tx (assoc data :metadata nil))
             (catch SQLException _
               (update-results tx data)))))
    nil))

(s/fdef update-with-results
        :args (s/cat :db-spec any?
                     :project-id ::specs/id
                     :results ::results)
        :ret nil?)

(defn insert-check
  "Insert PR check without the results yet"
  [db-spec project-id data metadata]
  (let [insertable (-> data
                       (assoc :coverage-results nil)
                       (assoc :project-id project-id)
                       (assoc :metadata (pr-str metadata))
                       (->db-keys))]
    (jdbc/with-db-transaction [tx db-spec]
      (if-let [{:keys [id]} (get-by-props tx project-id data)]
        (do (update-results tx insertable) id)
        (->> (insert-results tx insertable) vals first)))))

(s/fdef insert-check
        :args (s/cat :db-spec any?
                     :project-id ::specs/id
                     :data ::results
                     :metadata map?)
        :ret any?)

(defn- ->result
  [data]
  (-> data
      ->kebab
      (update :metadata edn/read-string)
      (update :coverage-results edn/read-string)))

(defn get-by-id
  [tx id]
  (some-> (get-check-by-id tx {:id id})
          ->result))

(defn get-results
  "Return results for a project in between dates"
  [tx project-id from to]
  (some->> (get-code-coverage-results tx (map-of-db project-id (str from " 00:00:00") (str to " 23:59:59")))
           (map ->result)))

(s/fdef get-results
        :args (s/cat :db-spec any?
                     :project-id ::specs/id
                     :from ::date
                     :to ::date)
        :ret any?)
