// @flow
import $ from 'jquery';

import * as modal from 'codescene-ui/src/modal';
import * as alerts from 'codescene-ui/src/alerts';
import * as tableSorter from 'codescene-ui/src/table-sorter';
import * as xhr from 'codescene-ui/src/browser/xhr';

function showCommitDetails(projectId : number, analysisId : number, repositoryName : string, commitHash : string) {
  const container = $('<div>').addClass('function-code');
  alerts.progress(container, 'Loading...');

  modal.show({
    wrapperClass: 'function-code-modal',
    title: repositoryName + '  ' + commitHash,
    content: container,
    size: 'lg'
  });

  getCommitDetails(projectId, analysisId, repositoryName, commitHash)
    .then(contents => {
      const code = $('<code>').text(contents);
      const pre = $('<pre>').append(code);
      container.html(pre);
    }).catch(err => {
      console.error(err.stack);
      alerts.error(container, 'Failed to show detailed commit info.');
    });
}

function renderActions(risk, projectId, analysisId) {
  return  $('<button>')
    .addClass('default-button')
    .addClass('icon')
    .attr('title', 'Show Commit Details')
    .html('<i class="fa fa-list"></i>')
    .on('click', () => {
      showCommitDetails(projectId, analysisId, risk.repository, risk.commit);
    });
}

function renderRisk(risk, projectId, analysisId) {
  var row = $('<tr class="risk-commit">');

  $('<td class="commit-name"></td>')
    .append($('<code>').addClass('commit-hash').text(risk.commit))
    .appendTo(row);

  $('<td class="risk-category"></td>')
    .append(risk.risk)
    .appendTo(row);

  $('<td class="commit-date"></td>')
    .append(risk.date)
    .appendTo(row);

  $('<td class="commit-repository"></td>')
    .append(risk.repository)
    .appendTo(row);

  $('<td class="commit-actions"></td>')
    .append(renderActions(risk, projectId, analysisId))
    .appendTo(row);

  return row;
}

function getCommitDetails(projectId : number, analysisId : number, repositoryName : string, commitHash : string) {
  const repoName : string = encodeURIComponent(repositoryName);
  const commit : string = encodeURIComponent(commitHash);
  const url = `${window.baseUrl}${projectId}/analyses/${analysisId}/project-management/commits/info.json?repo=${repoName}&commit=${commit}`;
  return xhr.json(url)
    .then(r => r.details);
}

export function renderTable(data : Array<any>, projectId : number, analysisId : number) {
  var commitRisks = $('<table class="table table-striped table-hover commit-risk-table"></table>');
  $('<thead></thead>')
    .append(`
    <tr>
      <th class="commit-name">Commit</th>
      <th class="risk-category">Risk Category</th>
      <th class="commit-date">Date</th>
      <th class="commit-repository">Git Repository</th>
      <th class="commit-actions"></th>
    </tr>
  `)
    .appendTo(commitRisks);

  const tBody = $('<tbody></tbody>');
  tBody.appendTo(commitRisks);

  $(data).each(function (i, risk) {
    tBody.append(renderRisk(risk, projectId, analysisId));
  });

  tableSorter.sort(commitRisks, {
    sortList: [[2, 1], [1, 1]],
    headers: {
      '0': {
        sorter: false
      },
      '4': {
        sorter: false
      }
    }
  });

  return commitRisks;
}

