(ns codescene.features.plugins.core
  "Provides the API for loading plugins and extension implemented by plugins.
   The idea is to keep everything related to plugin and java interop away from users,
   and to always return extensions as clojure protocol implementations.
   Thus, all interfaces in codescene.plugin are wrapped using such protocols."
  (:require [codescene.features.plugins.system-map-extension :as sme]
            [codescene.features.plugins.virtual-code-reviewer-extension :as vcr]
            [codescene.features.plugins.project-configuration-extension :as pce]
            [codescene.features.plugins.global-configuration-extension :as gce]
            [codescene.features.plugins.plugin-manager :as pm])
  (:import (com.codescene.plugin SystemMapExtensionPoint
                                 VirtualCodeReviewerExtensionPoint
                                 GlobalConfigurationExtensionPoint
                                 ProjectConfigurationExtensionPoint)))

(defn- plugin-ids [plugin-manager]
  (map :id (pm/get-plugins plugin-manager)))

(defn- has-extensions? [plugin-manager extension-point]
  (not (empty? (pm/get-extensions plugin-manager extension-point))))

(defn- wrap-java-extensions 
  "Encapsulates common parts in the public API methods below"
  [plugin-manager plugin-id context extension-point wrapper-fn]
  (->> (pm/get-extensions plugin-manager extension-point plugin-id)
       (map #(wrapper-fn plugin-id % context))
       (filter some?)
       (into [])))

;;;;;;; The public API starts

(defn global-configuration-extensions
  ([{:keys [plugin-manager] :as system} db]
   (->> (plugin-ids plugin-manager)
        (mapcat (partial global-configuration-extensions system db))
        (into [])))
  ([{:keys [plugin-manager encryptor] :as _system} db plugin-id]
   (let [context {:db        db
                  :encryptor encryptor}]
     (wrap-java-extensions plugin-manager plugin-id context
                           GlobalConfigurationExtensionPoint gce/->ConfigurationExtension))))

(defn has-global-configuration-extensions?
  [{:keys [plugin-manager] :as _system}]
  (has-extensions? plugin-manager GlobalConfigurationExtensionPoint))

(defn project-configuration-extensions
  ([{:keys [plugin-manager] :as system} db global-settings]
   (->> (plugin-ids plugin-manager)
        (mapcat (fn [plugin-id]
                  (project-configuration-extensions system db (get global-settings plugin-id) plugin-id)))
        (into [])))
  ([{:keys [plugin-manager encryptor] :as _system} db global-settings plugin-id]
  (let [context {:db              db
                 :encryptor       encryptor
                 :global-settings global-settings}]
     (wrap-java-extensions plugin-manager plugin-id context
                           ProjectConfigurationExtensionPoint pce/->ConfigurationExtension))))

(defn has-project-configuration-extensions? [{:keys [plugin-manager] :as _system}]
  (has-extensions? plugin-manager ProjectConfigurationExtensionPoint))

(defn custom-metrics-providers
  ([{:keys [plugin-manager] :as system} settings project]
   (->> (plugin-ids plugin-manager)
        (mapcat (partial custom-metrics-providers system settings project))
        (into [])))
  ([{:keys [plugin-manager encryptor] :as _system} settings {:keys [repo-paths] :as _project} plugin-id]
  (let [context {:encryptor  encryptor
                 :repo-paths repo-paths
                 :settings   (get settings plugin-id)}]
     (wrap-java-extensions plugin-manager plugin-id context
                           SystemMapExtensionPoint sme/->CustomMetricsProvider))))

(defn virtual-code-reviewer-extensions
  ([{:keys [plugin-manager] :as system} settings]
   (->> (plugin-ids plugin-manager)
        (mapcat (partial virtual-code-reviewer-extensions system settings))
        (into [])))
  ([{:keys [plugin-manager encryptor] :as _system} settings plugin-id]
   (let [context {:encryptor encryptor
                  :settings (get settings plugin-id)}]
     (wrap-java-extensions plugin-manager plugin-id context
                           VirtualCodeReviewerExtensionPoint vcr/->VirtualCodeReviewerExtension))))

(defn installed-plugins [{:keys [plugin-manager] :as _system}]
  (pm/get-plugins plugin-manager))

(comment
  (require '[cacsremoteservice.database.db :as db])
  (def db (db/persistent-connection))
  (init db)
  (shutdown)
  (let [plugin-id "clojure-plugin"]
    (-> (pm/get-extensions @plugin-manager GlobalConfigurationExtensionPoint plugin-id)
        first 
       (#(gce/->ConfigurationExtension db plugin-id % {})) ))
  (global-configuration-extensions db)
  (plugin-ids)
  (global-configuration-extensions db)
  (has-global-configuration-extensions?)
  (project-configuration-extensions db 1)
  (has-project-configuration-extensions?)
  (project->with-extensions-for-analysis db {:id 1})
  (has-project-configuration-extensions?)
  (project-configuration-extensions db 1))
