(ns codescene.features.menus
  (:require [clojure.string :as str]))

(def analysis-tree
  [{:key :dashboard
    :title "Dashboard"
    :children []}

   {:key :scope
    :title "Scope"
    :children [{:key :analysis-data
                :title "Analysis data"}
               {:key :system-trends
                :title "System trends"}]}

   {:key :goals
    :title "Goals"
    :children [{:key :hotspot-goals
                :title "Hotspot Goals"}]
    :alternate-children {:top-level-link []}}

   {:key :code
    :title "Code"
    :children [{:key :hotspots
                :title "Hotspots"}
               {:key :technical-debt
                :title "Technical Debt"
                :badge "New"}
               {:key :hotspot-code-health
                :title "Hotspot Code Health"}
               {:key :change-coupling
                :title "Change Coupling"}
               {:key :performed-refactorings
                :title "Performed Refactorings"}
               {:key :code-health-improved-notification
                :title "Code Health Improved"
                :remove-by-default true}
               {:key :code-health-decline-notification
                :title "Code Health Decline"
                :remove-by-default true}
               {:key :rising-hotspots-notification
                :title "Rising Hotspots"
                :remove-by-default true}
               {:key :missed-goals-notification
                :title "Missed goals"
                :remove-by-default true}
               {:key :prediction-code-health-decline-notification
                :title "Prediction: Code Health Decline"
                :remove-by-default true}
               {:key :code-health-in-rising-hotspot-notification
                :title "Code Health issues in Rising Hotspots"
                :remove-by-default true}
               {:key :complexity-warning-notification
                :title "Rising Complexity"
                :remove-by-default true}
               {:key :critical-code
                :title "Critical Code"}]}

   {:key :code-coverage
    :title "Code Coverage"
    :children []}

   {:key :architecture
    :title "Architecture"
    :children [{:key :system-health
                :title "System Health"}
               {:key :architecture-hotspots
                :title "Hotspots"}
               {:key :conways-law
                :title "Conway's Law"}
               {:key :architecture-change-coupling
                :title "Change Coupling"}]
    :alternate-children {:configure [{:key :configure-architectural
                                      :title "Configure"}]
                         :about [{:key :configure-architectural
                                  :title "About"}]}}

   {:key :team-dynamics
    :title "Team Dynamics"
    :children [{:key :delivery-effectiveness
                :title "Delivery Effectiveness"}
               {:key :team-code-alignment-explorer
                :title "Team-Code Alignment Explorer"}
               {:key :individuals
                :title "Individuals"}
               {:key :teams
                :title "Teams"}
               {:key :modus-operandi
                :title "Modus Operandi"}
               {:key :author-stats
                :title "Author Statistics"}
               {:key :team-stats
                :title "Team Statistics"}
               {:key :possible-ex-devs-notification
                :title "Possible Ex-Developers"
                :remove-by-default true}
               {:key :teamless-authors-notification
                :title "Teamless Authors"
                :remove-by-default true}]
    :alternate-children {:configure [{:key :configure-teams
                                      :title "Configure"}]}}

   {:key :system
    :title "System"
    :children [{:key :development-costs
                :title "Development Costs"}
               {:key :branches
                :title "Branches"}
               {:key :risk             ; to be deprecated
                :title "Risk"}
               {:key :pr-statistics
                :title "PR Statistics"}
               {:key :long-lived-branch-notification
                :title "Long Lived Branch"
                :remove-by-default true}
               {:key :high-risk-commits-notification
                :title "High Risk Commits"
                :remove-by-default true}]}

   {:key :delivery-performance
    :title "Delivery Performance"
    :children [{:key :delivery-performance-metrics
                :title "Metrics"}
               {:key :avg-development-times
                :title "Average development times"}]}

   {:key :simulations
    :title "Simulations"
    :children [{:key :offboarding-simulator
                :title "Offboarding"}]}

   {:key :project-configuration
    :title "Configuration"
    :children []}])

(def global-tree
  [{:key :global-configuration
    :title "Configure"
    :children [{:key :teams-developers-config
                :title "Teams/Developers"}
               {:key :reports-configuration
                :title "Reports"}
               {:key :plans
                :title "Plans"}
               {:key :instance-configuration
                :title "Configuration"}]}
   {:key :global-documentation
    :title "Learn"
    :children [{:key :tutorials
                :title "Tutorials"}
               {:key :documentation
                :title "Documentation"}
               {:key :help-center
                :target "_blank"
                :title "Help Center"}]
    :alternate-children {:cloud-anonymous [{:key :about
                                            :title "About"}
                                           {:key :faq
                                            :title "FAQ"}
                                           {:key :blog
                                            :title "Blog"}
                                           {:key :plans
                                            :title "Plans"}]}}

   {:key :logout
    :title "Logout"}])




(defn- make-remove-pred
  "`true` means remove this."
  [removes transforms]
  (let [remove-set (set removes)]
    (fn [{:keys [remove-by-default] item-key :key}]
      (or
        (remove-set item-key)
        (and remove-by-default
             (not (get-in transforms [item-key :override-remove-by-default])))))))

(defn- use-alternate-children
  [item alternate-children-kw]
  (if-let [new-children (get-in item [:alternate-children alternate-children-kw])]
    (assoc item :children new-children)
    ;; TODO: if we hit this, there's a problem: do we throw an exception, log, or fail gracefully?
    item))

(defn- apply-transforms
  [{item-key :key :as item} transforms]
  (if-let [{:keys [disable? alternate-children]}  (get transforms item-key)]
    (cond-> item
      disable? (assoc :disabled true)
      alternate-children (use-alternate-children alternate-children))
    item))

(defn- format-path
  "A path pattern is a vector containing one, two or three strinsg.
  The project id will be inserted
  after the first item in the tuple, and the analyis/job id after the
  second item. In On-prem, the first item should always be a `/`, and
  in Cloud it will always be `/project/`. We'll leave that logic to
  the caller though."
  [path {:keys [project-id analysis-id]}]
  (case (count path)
    1 (first path)
    2 (str (first path) project-id (second path))
    3 (str (first path) project-id (second path) analysis-id (nth path 2))
    (throw (ex-info "Invalid menu path subpath count" {:sub-path-count (count path)
                                                       :path path}))))

(defn- apply-path
  "Calculates the path for an item.

  Project-level paths generally only need the `project-id` and the
  `analysis-id` from the context. They can be defined using the
  `:path` key, which should be a vector of one, two or three items.

  Likewise, simple string paths can use `:path`, by supplying a vector
  containing only one string.

  For more complex cases, use `:path-fn` instead. It takes the entire
  `context` argument which should allow for lots of flexibility."
  [{item-key :key :as item} paths context]
  (let [{:keys [path path-fn]} (get paths item-key)]
    ;; Top-level items might not have a path at all, but still be
    ;; present in the `paths` map.
    (if (or path path-fn)
      (let [path-formatting-fn (or path-fn (partial format-path path))]
        (assoc item :path (path-formatting-fn context)))
      item)))

(defn- trim-glob-chars-from-match
  "Removes a trailing '/*' from the match string."
  [p]
  (str/replace p "/*" ""))

(defn- path-matches?
  [active-path matches context]
  (some
   (fn [m]
     (let [match (format-path m context)]
       (cond
         (= match active-path) true
         ;; NOTE: here we assume that '*' is always at  the end
         (not (str/ends-with? match "*")) false
         (str/starts-with? active-path (trim-glob-chars-from-match match)) true
         :else false)))
   matches))

(defn- decorate-active-path
  "To be applied after `apply-path`: appropriately decorates the item as
  active or not depending on the path match. This really only applies
  to sub-items because the `:active` and `:expanded` fields will be
  overwritten in `top-level-item`. "
  [{item-key :key item-path :path :as item} paths active-path context]
  (let [{:keys [matches]} (get paths item-key)]
    (if (or  (= item-path active-path)
             (path-matches? active-path matches context))
      (assoc item :active true :expanded false)
      (assoc item :active false :expanded false))))

(defn- decorate-item-fn
  "Returns a function that will be used for inserting all the fields
  into the menu items, both top-level and second-level items."
  [transforms paths context active-path]
  (fn [item]
    (-> item
        (apply-transforms transforms)
        (apply-path paths context)
        (decorate-active-path paths active-path context))))

(defn- top-level-item
  [remove-pred decorate {:keys [children active] :as decorated-item}]
  (if children
    (let [new-children (into [] (comp (remove remove-pred) (map decorate)) children)
          ;; The item might be active already, if it doesn't have any children
          active?  (boolean (or active (some :active new-children)))]
      (-> decorated-item
          (assoc
            :children new-children
            :active active?
            :expanded active?)
          (dissoc :alternate-children)))
    decorated-item))

(defn- path-without-query-part
  [active-path]
  (apply str (take-while #(not= % \?) active-path)))

(defn nav-menu
  "Constructs navigation menu tree by combining analysis-tree and global-tree (if supplied).
  Typically, you would use `analysis-tree` and `global-tree` provided by this ns,
  but custom trees are useful for tests."
  [analysis-tree global-tree {:keys [paths removes transforms context] :as _menu-info} active-path]
  (let [remove-pred (make-remove-pred removes transforms)
        decorate (decorate-item-fn transforms paths context (path-without-query-part active-path))
        xducer (comp
                (remove remove-pred)
                (map decorate)
                (map (partial top-level-item remove-pred decorate)))]
    [{:title "Navigation"
      :children
      (into [] xducer analysis-tree)}
     {:title "Global"
      :children (into [] xducer global-tree)}]))
(comment
  (nav-menu analysis-tree
            global-tree
            {:paths {:analysis-data {:path ["/" "/job/" "/the-data"]
                                     :matches []}}
             :removes []
             :transforms {}
             :context {:analysis-id 5 :project-id 1}}
            "/1/job/5/the-data")
  )

(defn match-path-to-menu-item
  "Matches `url-path` to a menu item descriptor as specified by `menu-description`.
  See menu definitions in onprem and cloud,
  e.g. `cacsremoteservice.views.analyses.menu-description/onprem-analysis-menu`.

  For more examples, check the tests: `codescene.features.menus-test/match-path-to-menu-item`"
  [menu-description url-path]
  ;; we remove any IDs in the URL path because they aren't used in menu definitions.
  ;; (see `codescene.features.menus.core/format-path` which adds them).
  ;; We do not need them because we supply no IDs in the 'context' arg for `path-matches?`."
  (let [normalize (fn [path] (str/replace path #"/\d+/" "//"))]
    (->> menu-description
         (keep (fn [[menu-key {:keys [path matches]}]]
            (when (path-matches? (normalize url-path)
                                 (into [path] matches)
                                 ;; no context/IDs needed because we normalized the path above
                                 {})
              menu-key)))
         first)))

(comment
  (match-path-to-menu-item (cacsremoteservice.views.analyses.menu-description/onprem-analysis-menu)
                           "/4/analyses/81/code/hotspots/system-map")
  ;; => :hotspots
  (match-path-to-menu-item (cacsremoteservice.views.analyses.menu-description/onprem-analysis-menu)
                           "/2/analyses/70/simulations/off-boarding")
  ;; => :offboarding-simulator
  )

