(ns codescene.features.delta.result-presenter
  "Contains fns used for presentation of delta result in the CodeScene UI"
  (:require [clojure.spec.alpha :as s]
            [clj-time.format :as ctf]
            [clojure.set :as set]
            [clojure.string :as str]
            [clj-time.coerce :refer [from-date]]
            [codescene.delta.delta-result :as analysis-delta-result]
            [codescene.features.delta.legacy-result-presenter :as legacy-result-presenter]
            [codescene.delta.specs :as delta-specs]
            [codescene.features.delta.quality-gates :as quality-gates]
            [codescene.features.delta.suppress :as suppress]

    ;; multi methods that we call:
            [hotspots-x-ray.recommendations.code-health.descriptions.code-duplication]
            [hotspots-x-ray.recommendations.code-health.descriptions.long-functions]
            [hotspots-x-ray.recommendations.code-health.descriptions.brain-method]
            [hotspots-x-ray.recommendations.code-health.descriptions.bumpy-roads]
            [hotspots-x-ray.recommendations.code-health.descriptions.cohesion-report]
            [hotspots-x-ray.recommendations.code-health.descriptions.complex-methods]
            [hotspots-x-ray.recommendations.code-health.descriptions.duplicated-assertions]
            [hotspots-x-ray.recommendations.code-health.descriptions.overall-code-complexity]
            [hotspots-x-ray.recommendations.code-health.descriptions.ctor-over-injection]
            [hotspots-x-ray.recommendations.code-health.descriptions.function-arguments]
            [hotspots-x-ray.recommendations.code-health.descriptions.large-assertion-blocks]
            [hotspots-x-ray.recommendations.code-health-interpretations :as chi]

    ;; ...and the multi method protocol:
            [hotspots-x-ray.recommendations.code-health.descriptions.delta-description-multi-protocol :as delta-description]
            [medley.core :as m]))

(defn- finding
[{:keys [name hotspot?] :as _file-result}
 {:keys [category threshold] :as _finding}
 {:keys [change-type suppression locations description change-level value] :as _change-details}]
(let [{why-it-occurs :description how-to-fix :fix} (delta-description/describe {:title category :value value :file-name name :threshold threshold :c _change-details})]
  (m/assoc-some
   {:name category
    :file name
    :change-type change-type
    :change-level change-level}
   :line (or (some :start-line locations) (some :start-line-before locations))
      ;; findings with multiple locations are file-level and thus shouldn't include a method name here
   :method (when (= 1 (count locations)) (-> locations first :function))
      ;;:impact 3
   :is-hotspot? hotspot?
      ;;:goal "Supervise"
   :why-it-occurs why-it-occurs
   :how-to-fix how-to-fix
      ;;:refactoring-examples


      ;;:what-has-improved ""
   :what-changed description

   :suppression suppression)))

(defn- with-counts [pred-fn findings]
  (let [findings (keep pred-fn findings)]
    {:number-of-types (count (into #{} (map :name) findings))
     :number-of-files-touched (count (into #{} (map :file) findings))
     :findings findings}))

(defn- add-refactoring-examples
  [get-refactorings-fn finding]
  (assoc finding :refactoring-examples
         (get-refactorings-fn (:name finding) (chi/filename->language (:file finding)))))

(defn suppressed-finding [{{:keys [reason suppressed-by updated-at]} :suppression :as finding}]
  (when (and updated-at (-> finding :change-type analysis-delta-result/negative-changes?))
    (assoc finding
      :reason reason
      :suppressed-by suppressed-by
      :suppressed-date (ctf/unparse (ctf/formatters :date-time)
                                    (from-date updated-at)))))

(defn filter-by-type [get-refactorings-fn change-level]
  (fn [finding]
    (when (= (:change-level finding) change-level)
      (if (= :improvement change-level)
        finding
        (when (not (:suppression finding))
          (add-refactoring-examples get-refactorings-fn finding))))))

(defn- affected-hotspots
  [{:keys [file-results] :as _result}]
  (->> file-results (filter :hotspot?) count))

(defn map-directives
  [directives]
  (map (fn [{:keys [rules fn-name]}]
         {:rules (str/join "; " rules)
          :fn-name fn-name})
       directives))

(defn ->directives [file-results]
  (reduce (fn [acc {:keys [directives name]}]
            (cond-> acc
              (seq (:added directives)) (update :added conj {:name name
                                                             :directives (map-directives (:added directives))})
              (seq (:removed directives)) (update :removed conj {:name name
                                                                 :directives (map-directives (:removed directives))})))
          {:added [] :removed []}
          file-results))

(defn- files-for-ui
  [file-scores pred]
  (let [scores (filter pred file-scores)]
    {:code-health (analysis-delta-result/weighted-score scores :score :loc)
     :old-code-health (analysis-delta-result/weighted-score scores :old-score :old-loc)
     :files (mapv #(set/rename-keys % {:score :code-health
                                       :old-score :old-code-health})
                  scores)}))

(defn negative-review? [{:keys [file-results code-health-alert-level]} suppressions]
  (-> (if (some :enabled-gates file-results)
        (:fail (quality-gates/qg-summary suppressions file-results))
        (analysis-delta-result/keep-findings (analysis-delta-result/negative-review-pred code-health-alert-level) suppressions file-results))
      first
      some?))

(defn transform-results* [{:keys [suppressions]} get-refactorings-fn {:keys [file-results] :as result}]
  (let [findings (analysis-delta-result/keep-findings finding suppressions file-results)
        notices (with-counts (filter-by-type get-refactorings-fn :notice) findings)
        negative-findings (with-counts (filter-by-type get-refactorings-fn :warning) findings)
        positive-findings (with-counts (filter-by-type get-refactorings-fn :improvement) findings)
        directives (->directives file-results)
        scores (analysis-delta-result/file-scores result)]
    {:negative-findings negative-findings
     :notices notices
     :suppressions (with-counts suppressed-finding findings)
     :positive-findings positive-findings
     :directives directives
     :is-negative-review (negative-review? result suppressions)
     :negative-impact-count (-> negative-findings :findings count)
     :positive-impact-count (-> positive-findings :findings count)
     :affected-hotspots (affected-hotspots result)
     :added-files (files-for-ui scores analysis-delta-result/new-score?)
     :removed-files (files-for-ui scores analysis-delta-result/deleted-score?)
     :modified-files (files-for-ui scores analysis-delta-result/modified-score?)
     :code-health (analysis-delta-result/weighted-score scores :score :loc)
     :old-code-health (analysis-delta-result/weighted-score scores :old-score :old-loc)}))

(s/def ::delta-result-scope (s/keys :req-un [::suppress/suppressions]))
(s/fdef transform-results
        :args (s/cat :delta-result-scope ::delta-result-scope
                     :get-refactorings-fn fn?
                     :result ::delta-specs/delta-analysis-result)
  :ret map?)
(defn transform-results [delta-scope get-refactorings-fn result]
  (-> (select-keys result [:commits :authors :analysis-time :external-review-id :repo :version])
      (merge (if-not (analysis-delta-result/legacy-version? result)
               (transform-results* delta-scope get-refactorings-fn result)
               (legacy-result-presenter/transform-results get-refactorings-fn result))
             {:external-review-provider (:external-review-provider delta-scope)
              :external-review-url (:external-review-url delta-scope)})))
