(ns codescene.features.delta.quality-gates
  (:require [camel-snake-kebab.extras :as csk-extras]
            [clojure.set :as set]
            [clojure.string :as str]
            [codescene.util.json :as json]
            [codescene.util.string :as u.str]
            [codescene.delta.detectors.quality-gates :as qg]
            [codescene.delta.delta-result :refer [xf-results change-details descope]]
            [codescene.features.config.pr-config :as pr-config]
            [codescene.features.config.presets :as presets]
            [evolutionary-metrics.mining.file-patterns :as file-patterns]
            [medley.core :as m]
            [taoensso.timbre :as log]))

(defn finding-function
  "Some findings refer to multiple functions, and those should return nil,
  otherwise return the function of the finding."
  [locations]
  (when (= 1 (count locations)) (-> locations first :function)))

(defn score-change [file-result]
  (let [os (:old-score file-result 10.00)
        s (:score file-result os)]
    (- s os)))

(defn score-change-str
  "If the file doesn't have an old score, we can't compare it to anything,
  in which case we just return the current score. This would be the case
  for a new file."
  [{:keys [old-score score]}]
  (cond
    (= score old-score)
    "no change"

    (nil? old-score)
    (format "%.2f" score)

    :else (format "%.2f → %.2f" old-score score)))

(defn description [{:keys [gate causes]} new-file-threshold]
  (let [categories-str (str/join ", " (change-details true (fn [_ {:keys [category]} _] category) causes))
        causes-str #(u.str/pluralize (count causes) %)]
    (case gate
      "hotspot-decline" (format "%s with %s" (causes-str "hotspot") categories-str)
      "new-code-health" (format "%s with code health below %.2f" (causes-str "new file") (double new-file-threshold))
      "refactoring-goals" (format "%s that weren't refactored" (causes-str "file"))
      "supervise-goals" (format "%s with %s" (causes-str "supervised file") categories-str)
      (format "%s with %s" (causes-str "file") categories-str))))

(defn cause-printed [{:keys [review] :as file-result} gate finding-url-fn provider-url-fn]
  (if (= "refactoring-goals" gate)
    {:file-name (:name file-result)}
    (let [reviews-str #(u.str/pluralize (count review) %)
          n (:name file-result)
          ;; one of the failed items, used to create suppression links and findings links
          {:keys [finding-url provider-url]}
          (first (change-details false
                                 (fn [_ {:keys [category]} {:keys [locations]}]
                                   {:finding-url (finding-url-fn category n (finding-function locations))
                                    :provider-url (provider-url-fn category n)})
                                 [file-result]))]
      {:file-name (file-patterns/final-segment n)
       :suppress-url (str finding-url "&suppress=true")
       :provider-url provider-url
       :violations (case gate
                     "hotspot-decline" (format "%s in this hotspot" (reviews-str "rule"))
                     "new-code-health" (reviews-str "rule")
                     "supervise-goals" (format "%s in this supervised file" (reviews-str "rule"))
                     "critical-health-rules" (reviews-str "critical rule")
                     "advisory-health-rules" (reviews-str "advisory rule"))
       :code-health-impact (score-change-str file-result)})))

(defn qg-settings-id
  "Returns ID of QG settings stored at scope"
  [db-spec scope]
  (or (get-in (presets/available-settings db-spec scope ["pr-quality-gates"]) ["pr-quality-gates" 0 :id])
      (presets/create-settings db-spec
                               scope
                               identity
                               {:category "pr-quality-gates"
                                :label "Quality Gates"
                                :qg-preset "minimal"})))

(defn qg-presets
  "Returns global or other scoped qg presets"
  [db-spec scope]
  (let [id (qg-settings-id db-spec scope)]
    (presets/get-settings db-spec id scope)))

(defn enabled-gates*
  [props global-config]
  (if (= "global" (:qg-preset props))
    (enabled-gates* (:kv global-config) nil)
    (let [qg-preset (:qg-preset props "minimal")
          custom-props (select-keys props pr-config/qg-properties-keys-wo-preset)]
      (merge (qg/profile qg-preset
                         (set/rename-keys custom-props
                                          {:qg-hotspot-decline? "hotspot-decline"
                                           :qg-new-code-health? "new-code-health"
                                           :qg-critical-health-rules? "critical-health-rules"
                                           :qg-advisory-health-rules? "advisory-health-rules"
                                           :qg-refactoring-goals? "refactoring-goals"
                                           :codeowners-for-critical-code? "tag-codeowners-for-critical-code"
                                           :qg-supervise-goals? "supervise-goals"}))
             {:qg-preset qg-preset}))))

(defn enabled-gates
  "Returns gates that are enabled given project props and the globals scope.
  
  Loads global gate settings and combines with project specific settings."
  [db-spec props scope]
  (enabled-gates* props (qg-presets db-spec scope)))

(defn completely-disabled-gates
  "Returns set of gates that are disabled. These are either disabled for all files in the
  results or they are disabled for all files with the gate failing in the result."
  [file-results]
  (reduce (fn [disabled-gates {:keys [enabled-gates]}]
            (reduce disj disabled-gates (keys (m/filter-vals true? enabled-gates))))
          (set (keys qg/gate-names))
          file-results))

(defn- status [causes has-suppressed? has-disabled?]
  ;; if there were causes, but they are all gone they have been either disabled by per-file rule
  ;; or suppressed, so if there were no suppressions applied, the causes have all been eliminated
  ;; by per-file disable rules
  (if (empty? causes)
    (if (and (not has-suppressed?)
             has-disabled?)
      :disabled :pass)
    :fail))

(defn- remove-fr-props [causes]
  (mapv #(dissoc % :code-properties :penalties) causes))

(defn- gather-fr-goal
  [gate file-results]
  (let [has-disabled? (atom false)
        causes (keep (fn [{:keys [enabled-gates qg-fail] :as fr}]
                       (when (some (partial = gate) qg-fail)
                         (if (enabled-gates gate)
                           (select-keys fr [:name])
                           (do (reset! has-disabled? true) nil))))
                     file-results)
        status (cond
                 (seq causes) :fail 
                 @has-disabled? :disabled
                 :else :pass)]
    (log/infof "Gate=%s has status=%s caused by files %s"
               gate status (str/join ", " (map :name causes)))
    {:gate gate
     :status status
     :causes (remove-fr-props causes)}))

(defn gather-cd-goal
  [gate suppressions file-results]
  (let [has-suppressed? (atom false)
        has-disabled? (atom false)
        do-cd (fn [{:keys [enabled-gates] :as fr}
                   {:keys [category]}
                   {:keys [locations qg-fail]}]
                (when (some #(= gate %) qg-fail)
                  (let [deactivated? (cond
                                       (not (enabled-gates gate)) 
                                       (reset! has-disabled? true)
                                       (some #(get-in suppressions [(:name fr) category %])
                                             (cons nil (map :function locations)))
                                       (reset! has-suppressed? true)
                                       :else
                                       false)]
                    (not deactivated?))))
        gate-causes (remove-fr-props (xf-results do-cd file-results))
        gate-status (status gate-causes @has-suppressed? @has-disabled?)]
    (log/infof "Gate=%s has status=%s caused by files %s"
               gate
               gate-status
               (str/join ", " (map :name gate-causes)))
    {:gate gate
     :status (status gate-causes @has-suppressed? @has-disabled?)
     :causes gate-causes}))

(defn- gather-gate [gate suppressions disabled-gates file-results]
  (cond
    (disabled-gates gate)
    {:gate gate
     :status :disabled
     :causes []}
    (= gate "refactoring-goals")
    (gather-fr-goal gate file-results)
    :else
    (gather-cd-goal gate suppressions file-results)))

(defn qg-summary
  "Returns gates with status and active, empty file results reports all gates as disabled."
  [suppressions file-results]
  (let [cleanup-fr #(-> %
                        (update :name descope)
                        (update :old-name descope)
                        ;; thank you, JSON
                        (update :enabled-gates (partial csk-extras/transform-keys name)))
        file-results (map cleanup-fr file-results)
        disabled-gates (completely-disabled-gates file-results)]
    (log/info "Gates disabled: " disabled-gates)
    (->> (keys (dissoc qg/gate-names "tag-codeowners-for-critical-code"))
         (mapv #(gather-gate % suppressions disabled-gates file-results))
         (group-by :status))))

(defn cc-codeowners-to-notify [file-results]
  (reduce (fn [ret {:keys [code-owners-for-cc enabled-gates]}]
            (if (get enabled-gates "tag-codeowners-for-critical-code")
              (into ret code-owners-for-cc)
              ret))
          (sorted-set)
          file-results))

(defn template-as-string []
  (json/generate-string (qg/generate-template) {:pretty true}))

(defn failed-gates
  [suppressions file-results]
  (let [{:keys [disabled fail pass]} (qg-summary suppressions file-results)
        with-state (fn [state gates] (->> gates (map (fn [{:keys [gate]}] [gate state]))))]
    (into {} (concat
              (with-state false disabled)
              (with-state false pass)
              (with-state true fail)))))

(defn enabled-gates-in
  [file-results]
  (->> file-results
       (map :enabled-gates)
       (apply merge-with #(or %1 %2) {})))
