(ns codescene.features.dashboard.core
  "Common code used to collect data used for the dashboard UI."
  (:require
    [clojure.java.io :as io]
    [codescene.analysis.paths :as paths]
    [codescene.dev-ops.multi-repo-summary :refer [read-release-trend-summary-for]]
    [codescene.features.warnings.warnings-presentation :as warnings-presentation]
    [codescene.util.csv-conversions :as csv-conversions]
    [hotspots-x-ray.recommendations.code-health.supported-languages :as supported-languages]
    [semantic-csv.core :as sc]
    [clojure.string :as str]
    [codescene.features.factors.diagnostics :as diagnostics]))


(defn commits-last-month [analysis-results-path]
  (let [csv-file (io/file analysis-results-path paths/dashboard-csv)
        dashboard-data (csv-conversions/optional-csv->map csv-file)]
    (:commitslastmonth dashboard-data)))


(defn- add-supported
  "Add a key that says whether this is a language supported by CodeScene."
  [language-summary]
  (assoc language-summary
         :supported
         (supported-languages/language-supported? (:language language-summary))))

(defn file-summary-per-language [analysis-results-path]
  (let [file (io/file analysis-results-path paths/file-summary-csv)]
    (if (.exists file)
      (->> (sc/slurp-csv
             file
             :cast-fns {:blank sc/->int
                        :code sc/->int
                        :comment sc/->int
                        :files sc/->int})
           (map add-supported))
      [])))

;; Dashboard widget data

(def ^:private code-health-dashboard-params
  "Keys for code health metrics. Prefixed with 'code-health-' in dashboard.csv."
  [:red-code
   :yellow-code
   :green-code
   :file-coupling-by-commits
   :file-coupling-across-commits
   :file-coupling-between-repos
   :architectural-components
   :architectural-coupling-by-commits
   :architectural-coupling-across-commits
   :components-improving
   :components-declining
   :pr-checks
   :total-negative-findings
   :total-suppressed-findings
   :total-ignored-findings])

(def ^:private file-level-hotspot-widget
  [:defectratioinhotspots
   :red-hotspot-percentage
   :red-hotspot-development-effort])

(def ^:private knowledge-distribution-dashboard-params
  "Keys for knowledge distribution metrics. Prefixed with 'knowledge-distribution-' in dashboard.csv."
  [:knowledge-islands
   :complex-knowledge-islands
   :complex-code-former-contributors
   :code-by-former-contributors
   :no-coordination-files
   :low-coordination-files
   :medium-coordination-files
   :high-coordination-files
   :low-health-hotspots])

(def ^:private team-code-alignment-dashboard-params
  "Keys for team code alignment metrics. Prefixed with 'team-code-alignment' in dashboard.csv."
  [:high-team-coordination-files
   :high-team-coordination-components
   :total-files
   :total-components])

(defn- try-parse-as-number
  [value]
  (try
    (Float/parseFloat value)
    (catch NumberFormatException _ value)))

(defn- dashboard-value
  "Gets a value from the map based on dashboard.csv. If the key does not exist
  the key name in returned. This is so the GUI can show what the missing key is."
  [dashboard-data param]
  (-> (get dashboard-data param (name param))
      try-parse-as-number))

(defn- prefix-key [prefix key]
  (keyword (str prefix "-" (name key))))

(defn- dashboard-map [dashboard-data prefix params]
  (into {} (map (fn [param]
                  [param (dashboard-value dashboard-data (prefix-key prefix param))])
                params)))

(defn- strip-key-prefix [strip full-key]
  (keyword (str/replace-first (str full-key) (str strip "-") "")))

(defn- map-keys [f m]
  (into {} (map (fn [[k v]] [(f k) v]) m)))

(defn- releases-for-period [devops-releases time-period]
  (let [for-period (->> devops-releases keys (filter #(str/starts-with? % (str time-period))))]
    (map-keys (partial strip-key-prefix time-period) (select-keys devops-releases for-period))))

(defn- release-trend-summary [analysis-results-path]
  (let [devops-releases (:releases (read-release-trend-summary-for {:analysis-path-fn (partial paths/make-analysis-path-to analysis-results-path)}))
        week (releases-for-period devops-releases :weekly)
        month (releases-for-period devops-releases :monthly)
        year (releases-for-period devops-releases :yearly)
        last (releases-for-period devops-releases :last)]
    {:week week
     :month month
     :year year
     :last last}))

(defn- work-types-from
  [work-trend-record]
  (let [work-types (filter #(not (= "date" %)) (keys work-trend-record))]
    (map (fn [work-type]
           {:name work-type
            :minutes (try-parse-as-number (get work-trend-record work-type))})
         work-types)))

(defn- system-level-work-types
  [analysis-results-path]
  (let [f (io/file analysis-results-path paths/system-costs-trend-path-name paths/system-costs-trend-syntetic-component-name  paths/work-trend-file-name)]
    (when (.exists f)
      (map (fn [trend-record]
             {:date (get trend-record "date")
              :work-types (work-types-from trend-record)})
           (sc/slurp-csv f :keyify false)))))

(defn- author-stats
  [dashboard-data]
  (-> dashboard-data
      (select-keys [:authors
                    :activeauthors
                    :mincontribtime
                    :firstquartilecontribtime
                    :mediancontribtime
                    :thirdquartilecontribtime
                    :maxcontribtime
                    :exdevelopersmedianexperience
                    :nkeypersonnelpercentage
                    :nkeypersonnel
                    :knowledgelosspercentage])
      (update-vals try-parse-as-number)))

(defn- author-experience
  [analysis-results-path]
  (let [f (io/file analysis-results-path paths/author-experience-trend-csv)]
    (when (.exists f)
      (-> (sc/slurp-csv f)
          last
          (update-vals try-parse-as-number)))))

(comment
  {:date "2022-01-01"
   :work-types [{:name "Cloud" :hours 22}
                {:name "On-prem" :hours 24}]}
  (work-types-from {"date" "2021-03", "Cloud" "3.0", "On-Prem" "2.0"}))

(defn application-code-loc
  "Returns number of application code lines."
  [dashboard-csv-file]
  (let [dashboard-data (csv-conversions/optional-csv->map dashboard-csv-file)]
    ;; `:application-code-loc` ignores content that is excluded from analysis
    (or (some-> dashboard-data :application-code-loc Long/parseLong)
        ;; fallback for older analyses
        (some-> dashboard-data :codesize (str/replace #"[.,]" "") Long/parseLong)
        0)))

(defn dashboard-data
  "Get an aggregated dashboard data suitable for rendering KPI widgets, containing data for all four factors.
  The args are a full path to an anlysis result, and the four factors kpi-time-series."
  [analysis-results-path]
  (let [csv-file (io/file analysis-results-path paths/dashboard-csv)
        dashboard-data (csv-conversions/optional-csv->map csv-file)]
    {:code-health            (merge (dashboard-map dashboard-data "code-health" code-health-dashboard-params)
                                    (select-keys dashboard-data file-level-hotspot-widget))
     :team-code-alignment    (merge (dashboard-map dashboard-data "team-code-alignment" team-code-alignment-dashboard-params)
                                    (select-keys dashboard-data [:teams]))
     :knowledge-distribution (merge (dashboard-map dashboard-data "knowledge-distribution" knowledge-distribution-dashboard-params)
                                    (author-stats dashboard-data)
                                    {:experience (author-experience analysis-results-path)})
     :delivery-performance   (merge (select-keys dashboard-data [:branchttl :nbranches :branchmergelead])
                                    {:release-trend-summary (release-trend-summary analysis-results-path)}
                                    {:system-level-work-types (last (system-level-work-types analysis-results-path))})

     :early-warnings (warnings-presentation/get-supported-early-warnings analysis-results-path)
     :config-diagnostics (diagnostics/diagnose analysis-results-path)
     :application-code-loc (application-code-loc csv-file)}))

(comment
  (def analysis-results-path "/Users/jlindbergh/Downloads/analyses/cs_prod/analysis20230308181501")
  (def dash-from-csv (csv-conversions/optional-csv->map (io/file analysis-results-path paths/dashboard-csv)))

  (merge (dashboard-map dash-from-csv "knowledge-distribution" knowledge-distribution-dashboard-params)
         (author-stats dash-from-csv)
         {:experience (author-experience analysis-results-path)})


  (author-stats dash-from-csv)
  (author-experience analysis-results-path)
  (dashboard-map dash-from-csv "code-health" code-health-dashboard-params)
  (merge (select-keys dash-from-csv [:branchttl :nbranches :branchmergelead])
         {:release-trend-summary (release-trend-summary analysis-results-path)}
         {:system-level-work-types (last (system-level-work-types analysis-results-path))}))
