(ns codescene.features.components.project
  (:require [clojure.spec.alpha :as s]
            [codescene.features.spec.commons :as commons-spec]
            [codescene.features.api.spec.project :as project-spec]
            [codescene.features.spec.auth :as auth]))

(s/def ::pattern string?)
(s/def ::component string?)
(s/def ::architectural-component (s/keys :req-un [::pattern ::component]))
(s/def ::repo-paths (s/coll-of string?))
(s/def ::developer-configuration ::commons-spec/id)
(s/def ::config any?)
(s/def ::code-health boolean?)
(s/def ::missed-goals boolean?)
(s/def ::system-mastery boolean?)
(s/def ::badges (s/keys :opt-un [::code-health ::missed-goals ::system-mastery]))
(s/def ::urls (s/coll-of string?))
(s/def ::url string?)
(s/def ::path string?)
(s/def ::branch (s/nilable string?))
(s/def ::repository (s/keys :req-un [::url] :opt-un [::branch]))
(s/def ::repo_path (s/keys :req-un [::path] :opt-un [::commons-spec/id ::branch]))
(s/def ::repositories (s/coll-of ::repository))

(defprotocol ProjectComponent
  (-all-accessible-projects [this user])
  (-project-by-id [this project-id])
  (-project-config [this project-id] "Returns ProjectConfiguration")
  (-architectural-components [this project-id])
  (-update-architectural-components [this project-id data])
  (-create-new-project [this user project-spec])
  (-delete-project [this user project-id settings?])
  (-badge-status [this project-id])
  (-badge-status-update [this project-id params])
  (-all-repositories [this project-id])
  (-add-repositories [this project-id repositories])
  (-edit-repositories [this project-id repositories])
  (-remove-repositories [this project-id urls])
  (-run-analysis [this user project-id]))

(s/fdef all-accessible-projects
        :args (s/cat :this some?
                     :user ::auth/user)
        :ret (s/coll-of (s/keys :req-un [::commons-spec/id ::commons-spec/name])))
(defn all-accessible-projects
  "return all accessible projects for the given user"
  [this user]
  (-all-accessible-projects this user))

(s/fdef project-by-id
        :args (s/cat :this some?
                     :project-id ::commons-spec/id)
        :ret (s/nilable (s/keys :req-un [::commons-spec/name])))
(defn project-by-id
  "return the project for given id"
  [this project-id]
  (-project-by-id this project-id))

(s/fdef architectural-components
        :args (s/cat :this some?
                     :project-id ::commons-spec/id)
        :ret (s/coll-of ::architectural-component))
(defn architectural-components
  "return the architectural components list for the given project id"
  [this project-id]
  (-architectural-components this project-id))

(s/fdef update-architectural-components
        :args (s/cat :this some?
                     :project-id ::commons-spec/id
                     :data (s/coll-of ::project-spec/component-config)))
(defn update-architectural-components
  "replace existing architectural-components config with data"
  [this project-id data]
  (-update-architectural-components this project-id data))

(s/fdef create-new-project
        :args (s/cat :this some?
                     :user ::auth/user
                     :project-spec (s/keys :req-un [::config]))
        :ret (s/keys :req-un [::commons-spec/id ::commons-spec/name]))
(defn create-new-project
  "create new project using spec"
  [this user project-spec]
  (-create-new-project this user project-spec))

(s/fdef delete-project
        :args (s/cat :this some?
                     :user ::auth/user
                     :project-id ::commons-spec/id
                     :settings? boolean?)
        :ret any?)
(defn delete-project
  "delete project using id, if settings? is true also the developer settings will be deleted"
  [this user project-id settings?]
  (-delete-project this user project-id settings?))

(s/fdef badge-status
        :args (s/cat :this some?
                     :project-id ::commons-spec/id)
        :ret ::badges)
(defn badge-status
  "return the badges status for project with id"
  [this project-id]
  (-badge-status this project-id))

(s/fdef badge-status-update
        :args (s/cat :this some?
                     :project-id ::commons-spec/id
                     :params (s/coll-of string?))
        :ret any?)
(defn badge-status-update
  "update the badges status for project with id"
  [this project-id params]
  (-badge-status-update this project-id params))

(s/fdef all-repositories
        :args (s/cat :this some?
                     :project-id ::commons-spec/id)
        :ret ::repositories)
(defn all-repositories
  "all repositories of project with id"
  [this project-id]
  (-all-repositories this project-id))

(s/fdef add-repositories
        :args (s/cat :this some?
                     :project-id ::commons-spec/id
                     :repositories (s/keys :req-un [::repositories]))
        :ret (s/coll-of ::repo_path))
(defn add-repositories
  "add repositories to project with id"
  [this project-id repositories]
  (-add-repositories this project-id repositories))

(s/fdef edit-repositories
        :args (s/cat :this some?
                     :project-id ::commons-spec/id
                     :repositories (s/keys :req-un [::repositories]))
        :ret (s/coll-of ::repo_path))
(defn edit-repositories
  "edit repositories on project with id"
  [this project-id repositories]
  (-edit-repositories this project-id repositories))

(s/fdef remove-repositories
        :args (s/cat :this some?
                     :project-id ::commons-spec/id
                     :urls (s/keys :req-un [::urls]))
        :ret (s/coll-of string?))
(defn remove-repositories
  "remove repositories from project with id"
  [this project-id urls]
  (-remove-repositories this project-id urls))

(s/fdef run-analysis
  :args (s/cat :this some?
               :user ::auth/user
               :project-id ::commons-spec/id)
  :ret any?)
(defn run-analysis
  "run analysis for project with id"
  [this user project-id]
  (-run-analysis this user project-id))
