// @flow
import d3 from 'd3';
import $ from 'jquery';

import * as xhr from 'codescene-ui/src/browser/xhr';
import * as codeChurn from 'codescene-ui/src/analyses/code-churn';
import * as dateTrend from 'codescene-ui/src/analyses/date-trend';
import * as absoluteChurn from 'codescene-ui/src/analyses/absolute-churn';
import * as alerts from 'codescene-ui/src/alerts';

type ChurnTrendViewOptions = {
  churnTrendUrl : string,
  revisionTrendUrl : string,
  linesOfCodeTrendUrl : string,
  absoluteChurnTrendUrl : string,
  authorsPerMonthTrendUrl : string
};

export function init(options : ChurnTrendViewOptions) {
  // We want to use the same scale on all diagrams:
  function defaultYaxisHeight(d) {
    return Math.max(parseInt(d.rollingdeleted), parseInt(d.rollingadded));
  }

  function drawCodeChurnGraphs(yaxisHeightFn) {
    codeChurn.drawChurnFor(
      options.churnTrendUrl,
      '#addedchurn',
      'areaadded',
      (d) => d.rollingadded,
      (d) => { d.rollingadded = +d.rollingadded;},
      () => 0,
      yaxisHeightFn);
    codeChurn.drawChurnFor(
      options.churnTrendUrl,
      '#deletedchurn',
      'areadeleted',
      (d) => d.rollingdeleted,
      (d) => { d.rollingdeleted = +d.rollingdeleted;},
      () => 0,
      yaxisHeightFn);
  }

  function resetMaxChurnValue() {
    d3.select('#addedchurn svg').remove();
    d3.select('#deletedchurn svg').remove();

    drawCodeChurnGraphs(defaultYaxisHeight);
  }

  var maxChurnForm = $('#max-churn-value-form');
  var maxChurnValue = $('#max-churn-value');

  $('[name=reset-max-churn]').on('click', resetMaxChurnValue);

  function onChurnValueChanged(e) {
    e.preventDefault();

    var maxChurnToDisplay = parseInt(maxChurnValue.val());

    if(isNaN(maxChurnToDisplay) || maxChurnToDisplay < 1) {
      resetMaxChurnValue();
    } else {
      d3.select('#addedchurn svg').remove();
      d3.select('#deletedchurn svg').remove();

      drawCodeChurnGraphs(() => maxChurnToDisplay);
    }
  }

  maxChurnValue.on('change', () => {
    maxChurnForm.submit();
  });
  maxChurnForm.on('submit', onChurnValueChanged);

  xhr.csv(options.revisionTrendUrl).then(data => {
    dateTrend.draw($('#revisionschurn'), data, { renderAxis: true });
  });

  xhr.csv(options.authorsPerMonthTrendUrl).then(activeAuthors => {
    if (activeAuthors === undefined || activeAuthors.length == 0) {
      const trendContainer = $('#active-contributors-trend');
      alerts.warning(trendContainer, '<strong>Result not available.</strong> This is a new type of analysis so you need to re-run your analysis to get the results.');
    } else {

      dateTrend.drawAuthorTrend($('#active-contributors-trend'), activeAuthors, { renderAxis: true });
    }
  });

  function codeGrowthValue(d) {
    return parseInt(d.size);
  }
  codeChurn.drawChurnFor(
    options.linesOfCodeTrendUrl,
    '#modifiedchurn',
    'areamodified',
    (d) => d.size,
    (d) => { d.size = +d.size;},
    () => 0,
    codeGrowthValue);

  absoluteChurn.draw(options.absoluteChurnTrendUrl, '#absolutechurn');

  drawCodeChurnGraphs(defaultYaxisHeight);
}
