;; Tool to export the  content of H2 database as SQL file to be
;; imported into mysql/maridb using command line client
;; mysql --default-character-set=utf8mb4 -u <user> -p <dbname> < <export-file>
(ns tools.h2-to-mysql
  (:gen-class)
  (:require [clojure.java.jdbc :as jdbc]
            [clj-time.coerce :as c]
            [clj-time.format :as f]
            [clojure.java.io :as io]
            [clojure.set :as set]
            [clojure.string :as str])
  (:import (org.h2.jdbc JdbcClob)))

(defn- timestamp-to-string
  "convert java.sql.Timestamp to string parsable by mysql/mariadb"
  [date]
  (some->> date
           c/from-sql-time
           (f/unparse (f/formatter "yyyy-MM-dd HH:mm:ss"))))

(defn- date-to-string
  "convert java.sql.Date to string parsable by mysql/mariadb"
  [date]
  (some->> date
           c/from-sql-date
           (f/unparse (f/formatter "yyyy-MM-dd"))))

(defn- mysql-escape
  "escape string values for the SQL output file"
  [value]
  (some-> value
          (.replace "\\" "\\\\")
          (.replace "'", "\\'")
          (.replace "\"", "\\\"")))

(defn- single-quote
  "single quote the string value"
  [value]
  (when value
    (str "'" value "'")))

(defn hexify
  "convert array of bytes to hext string"
  [bytes]
  (apply str
         (map #(format "%02x" (int %)) bytes)))

(defn- transform
  "transform column value based on its type"
  [sqltype value]
  (cond
    (nil? value) "NULL"
    (= sqltype "timestamp") (-> (timestamp-to-string value) single-quote)
    (= sqltype "date") (-> (date-to-string value) single-quote)
    (= sqltype "character varying") (-> (mysql-escape value) single-quote)
    (= sqltype "character large object") (-> (mysql-escape value) single-quote)
    (= sqltype "boolean") (if value 1 0)
    :else value))

(defn- h2-db [db-path] {:db-type     "h2"
                        :class-name  "org.h2.Driver"
                        :subprotocol "h2"
                        :subname     (format "%s;MODE=MYSQL" db-path)
                        :user        "sa"
                        :password    ""})

(defn- declob
  [value]
  (if (instance? JdbcClob value)
    (slurp (.getCharacterStream value))
    value))

(defn get-tables
  "get all tables from public schema using INFORMATION_SCHEMA"
  [db-spec]
  (->> (jdbc/query db-spec
                   ["SELECT * FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = 'PUBLIC'"])
       (map :table_name)))

(defn get-table-columns
  "get all columns for a table from public schema using INFORMATION_SCHEMA"
  [db-spec table-name]
  (->> (jdbc/query db-spec
                   [(format "SELECT * FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = 'PUBLIC' AND TABLE_NAME = '%s'" table-name)])
       (map #(select-keys % [:column_name :data_type]))
       (map #(->> (map (fn [[k v]] {k (str/lower-case v)}) %) (into {})))))


(defn- sql-prefix
  [table-name]
  (str "INSERT INTO " table-name " VALUES ("))

(defn- sql-postfix
  [columns]
  (if (not-empty (rest columns)) ", " ");\n"))

(defn- row-fn
  [row]
  (into {} (map (fn [[k v]] {k (declob v)}) row)))

(defn- table_cross_references
  "fetch table cross-references from INFORMATION_SCHEMA related to PK-FK"
  [db-spec]
  (->> (jdbc/query db-spec
                   [(str "SELECT tc.TABLE_NAME fktable_name, ccu.TABLE_NAME pktable_name "
                         " FROM INFORMATION_SCHEMA.TABLE_CONSTRAINTS TC, INFORMATION_SCHEMA.CONSTRAINT_COLUMN_USAGE CCU "
                         " WHERE TC.CONSTRAINT_TYPE = 'FOREIGN KEY' and TC.TABLE_SCHEMA = 'PUBLIC' and "
                         " tc.CONSTRAINT_NAME = ccu.CONSTRAINT_NAME and tc.TABLE_NAME != ccu.TABLE_NAME")])
       (map #(select-keys % [:pktable_name :fktable_name]))))

(defn- all-tables-with-order
  [db-spec]
  (let [cross-references (table_cross_references db-spec)
        pktables (->> cross-references (map :pktable_name) set)
        fktables (->> cross-references (map :fktable_name) set)
        union-pk-fk (set/union pktables fktables)
        not-in-fktables (set/difference pktables fktables)
        not-in-pktables (set/difference fktables pktables)
        order-fn (fn [table]
                   (cond
                     (not (contains? union-pk-fk table)) 1
                     (contains? not-in-fktables table) 2
                     (contains? not-in-pktables table) 999
                     :else 0))]
    (->> (get-tables db-spec)
         (map #(hash-map :table % :order (order-fn %))))))

(defn- pk-for-fk
  "get a list of all pk table where the given table is fk"
  [cross-references table]
  (->> cross-references
       (filter #(= table (:fktable_name %)))
       (map :pktable_name)
       set))

(defn- compute-order
  [tables cross-references table]
  (let [pk-tables (pk-for-fk cross-references table)
        orders (->> tables
                    (filter #(contains? pk-tables (:table %)))
                    (map :order)
                    sort)
        can-compute (= 0 (first orders))]
    (if can-compute 0 (-> orders last inc))))

(defn- update-table-order
  [tables cross-references]
  (let [order-fn (fn [{:keys [order table] :as t}]
                   (if (= order 0)
                     (->> (compute-order tables cross-references table)
                          (assoc t :order))
                     t))]
    (map order-fn tables)))

(defn schema-order
  "generate the order of insert to not break the PK-FK relationship"
  [db-spec]
  (let [cross-references (table_cross_references db-spec)]
    (loop [tables (all-tables-with-order db-spec)]
      (let [updated-tables (update-table-order tables cross-references)
            order-computed (->> updated-tables
                                (filter #(= 0 (:order %)))
                                empty?)]
        (if order-computed
          updated-tables
          (recur updated-tables))))))

(defn- to-SQL
  "generate insert SQL statement for a data item"
  [table-name metadata item]
  (loop [columns metadata sql (sql-prefix table-name)]
    (let [{:keys [column_name data_type]} (first columns)]
      (if column_name
        (let [transformed-value (->> (get item (keyword column_name))
                                     (transform data_type))
              computed-sql (str sql transformed-value (sql-postfix columns))]
          (recur (rest columns) computed-sql))
        sql))))

(defn get-table-data
  "fetch data from source database"
  [db-spec table-name]
  (let [metadata (get-table-columns db-spec table-name)]
    (->> (jdbc/query
           db-spec
           [(format "SELECT * FROM %s" table-name)]
           {:row-fn row-fn})
         (map #(to-SQL table-name metadata %)))))

(defn print-help
  []
  (println "h2-to-mysql.sh -h2 <path/db-file> -out <export-file>")
  (println "\t\t- ussualy the H2 database have extension .mv.db, \n\t\t  the <path/db-file> must not contain the extension")
  (println "\t\t- the content of <export-file> can be imported into mysql/maridb using:\n\t\t  mysql --default-character-set=utf8mb4 -u <user> -p <dbname> < <export-file>"))

(defn- write-delete-statements
  "tables is expected to be sorted in ascending alphabetically order"
  [w tables]
  (doseq [t (reverse tables)]
    (.write w (format "delete from %s;\n" t))))

(defn- write-insert-statements
  "tables is expected to be sorted in ascending alphabetically order"
  [w db-spec tables]
  (doseq [t tables]
    (doseq [row (get-table-data db-spec t)]
      (.write w row))))

(defn -main
  ([]
   (print-help))
  ([h2opt h2-db-file outopt export-file & _args]
   (if (and h2opt h2-db-file outopt export-file)
     (let [db-spec (h2-db h2-db-file)
         ordered-tables (->> (sort-by :order (schema-order db-spec))
                             (sort-by :order)
                             (map :table))]
     (with-open [w (io/writer export-file)]
       (write-delete-statements w ordered-tables)
       (write-insert-statements w db-spec ordered-tables)))
     (print-help))))
