grammar SwiftNestedComplexity;
options {superClass=hotspots_x_ray.languages.InterruptibleParser;}

// This grammar detects islands of nested complexity.
// Note that in Swift, curly braces are mandatory around loops and conditionals.

method
:
	expression* EOF
;

expression : complexity
           | block_expression
           | anything;

complexity: if_clause
          | else_clause
          | switch_clause
          | for_each_loop
          | while_loop
          | repeat_while_loop;

block_expression: OPENING_BRACE expression*? CLOSING_BRACE;

anything: ~(OPENING_BRACE | CLOSING_BRACE);

if_clause: IF some_condition multi_line_conditional_expression;
else_clause: ELSE multi_line_conditional_expression;

switch_clause: SWITCH some_condition multi_line_conditional_expression;

multi_line_conditional_expression: OPENING_BRACE expression*? CLOSING_BRACE;

for_each_loop: FOR for_each_loop_part IN for_loop_condition multi_line_conditional_expression;
for_each_loop_part:  ~(IN | OPENING_BRACE)+?;
for_loop_condition: conditions*?;

while_loop: WHILE some_condition multi_line_conditional_expression;
repeat_while_loop: REPEAT multi_line_conditional_expression WHILE some_condition;

some_condition: conditions+?;
conditions : conditional_operator
           | ~(OPENING_BRACE)
           | LEFT_PAREN conditions*? RIGHT_PAREN;

conditional_operator: OPERATORS;

fragment ESCAPED : '\\\\' | '\\"';
LITERAL : '"' ( ESCAPED | ~('\n'|'\r') )*? '"';

SINGLE_LITERAL : '\''  ~('\n'|'\r')*? '\'';

BACK_QUOTED : '`'  ~('\n'|'\r')*? '`';

LITERAL_CHAR : '\'' . '\'' -> skip;

IF: 'if';
ELSE: 'else';
FOR: 'for';
IN: 'in';

REPEAT: 'repeat';
WHILE: 'while';

SWITCH: 'switch';
CATCH: 'catch';
THROW: 'throw';
RETURN: 'return';

OPERATORS: '&&' | '||';

LEFT_PAREN: '(';
RIGHT_PAREN: ')';
OPENING_BRACE: '{';
CLOSING_BRACE: '}';
COLON: ':';
SEMICOLON: ';';

Whitespace : [ \t]+ -> skip;

BlockComment: '/*' .*? '*/' -> skip;
LineComment: '//' ~[\r\n]* -> skip;

NEWLINE : '\r'? '\n' -> skip;

ID : [a-zA-Z_][a-zA-Z0-9_]*;

ANY_CHAR : .; // Put this lexer rule last to give it the lowest precedence
