// This is the ECMA script grammar with additions necessary to parse TypeScript too.
// We've chosen to combine these two grammars since TypeScript is a strict superset of ECMA.

grammar ECMAScript;

translationunit
:
	expression* EOF
;

expression : test_declaration
           | function_declaration
           | class_block
           | top_level_block_statement
           | anything;

function_level_expressions: block_expression* EOF;

// Fallback: in case we fail to parse a leading class declaration, we fall back and
// try to detect functions inside what our parser considers a "block".
block_expression : test_declaration
                 | function_declaration
                 | member_function
                 | top_level_block_statement
                 | anything;

anything : ~(LeftBrace | RightBrace);

class_block: CLASS class_name anything*? LeftBrace class_body*? RightBrace;

class_name: ID;

class_body: member_function
          | class_block_statement
          | anything;

member_function: member_function_name function_scope
               | member_function_name '=' fat_arrow_function;

member_function_name: 'get' ID
                    | 'set' ID
                    | 'get'
                    | 'set'
                    | NEW
                    | ID;

class_block_statement : LeftBrace class_block_statement_content*? RightBrace;
class_block_statement_content: class_block_statement | anything;

top_level_block_statement : LeftBrace block_expression*? RightBrace;

test_declaration: test_description
                | general_bdd_test_declaration
                | jasmine_test_declaration
                | test_setup_tear_down;

test_description: DESCRIBE LeftParen test_description_name ',' fat_arrow_function  RightParen SEMICOLON?;

general_bdd_test_declaration: test_declarator LeftParen test_name ',' fat_arrow_function  RightParen SEMICOLON?;
jasmine_test_declaration: IT LeftParen test_name ',' lambda_function  RightParen;
test_declarator: TEST | IT;
test_name: LITERAL | SINGLE_LITERAL;
test_description_name: LITERAL | SINGLE_LITERAL;

test_setup_tear_down: setup_tear_down_name LeftParen test_setup_tear_down_body RightParen;
setup_tear_down_name: BeforeEach | AfterEach;
test_setup_tear_down_body: (fat_arrow_function | lambda_function);

function_declaration : property_label? ASYNC? FUNCTION function_name generic_type? function_scope
                     | function_name ASSIGN ASYNC? FUNCTION function_scope
                     | function_name ASSIGN fat_arrow_function
                     | EXPORT function_name generic_type? ASYNC? FUNCTION function_scope
                     | function_name COLON ASYNC? FUNCTION function_scope
                     | function_name COLON fat_arrow_function
                     | function_name function_scope
                     | function_name '=' function_decorator LeftParen wrapped_function_body RightParen // const myFn = memo((props: IProps) => {
                     | callback_function_name ',' fat_arrow_function // electron:  app.on("window-all-closed", () => {
                     | function_name '=' generic_type fat_arrow_function; // const identity = < T extends {} >(arg: T): T => { return arg; }

// We want to distinguish between plain functions (e.g. function a() ...) and
// functions using pre-ES6 object style (e.g. myMethod: function a() ...).
property_label: ID COLON;

function_decorator: ID
                  | SCOPED_NAME; // e.g. React.memo(...

function_name: ID
             | SCOPED_NAME
             | ID return_type
             | LITERAL
             | SINGLE_LITERAL;
callback_function_name: LITERAL | SINGLE_LITERAL;

lambda_function: ASYNC? FUNCTION function_scope;

wrapped_function_body_part: fat_arrow_function | top_level_block_statement;
wrapped_function_body_parts: wrapped_function_body_part
                           | wrapped_function_body_parts ',' wrapped_function_body_part;
wrapped_function_body: wrapped_function_body_parts+;

fat_arrow_function: ASYNC? fat_arrow_arg_list return_type? FAT_ARROW code_body_block;
fat_arrow_arg_list: ID
                  | LeftParen function_definition_params_list? RightParen;

// The code body block is part of fat arrow functions. This is where it gets complicated:
// A fat arrow can be curried:
//
// handleChange = field => e => {
//     e.preventDefault();
// }
//
// We handle that via the last recursive parse rule:
code_body_block: LeftBrace function_body RightBrace
                | SQUARE_LEFT function_body SQUARE_RIGHT
                | LeftParen LeftBrace function_body RightBrace RightParen
                | LeftParen function_body RightParen
                | new_object_creation LeftParen fat_arrow_lambda RightParen
                | fat_arrow_function;

// NOTE: this is not really a proper solution; here we just catch one thing, although it's a common
// construct in TypeScript:
//   export const dbConnect = (): Promise<{ success: boolean }> =>
//      new Promise(resolve => {
new_object_creation: NEW ID;
fat_arrow_lambda: function_definition_params_list FAT_ARROW code_body_block;

function_scope: LeftParen function_definition_params_list? RightParen return_type? LeftBrace function_body RightBrace;

return_type: COLON typescript_type;

// Bugfix: functions might end the last argument with a superfluous comma. Make sure to parse that one too:
function_definition_params_list : rec_function_definition_params_list ','?;

rec_function_definition_params_list : function_param
                                    | rec_function_definition_params_list (',' | ';') function_param
                                    ;

function_param: param_names
              | param_names '=' ~(',' | RightParen)+?
              | param_names QUESTION_MARK? COLON type_spec default_arg_value?
              | 'new()' QUESTION_MARK? COLON type_spec default_arg_value? ';'?
              | '...' function_param // rest arguments
              | ~('=' |':' | LeftBrace | '&&' | '||' | '==' | '===' )+? '=' ~LeftParen*? LeftParen RightParen
              | param_names COLON hof_type;

param_names: DOLLAR? ID
           | DOLLAR
           | destructuring_param;

destructuring_param: LeftBrace ~(RightBrace)+? RightBrace;

hof_type: type_spec FAT_ARROW type_spec;

default_arg_value: '=' .;

type_spec: typescript_type
         | LeftBrace rec_function_definition_params_list RightBrace
         | LeftParen rec_function_definition_params_list RightParen;

// Parse types like: 'Fiber | null' or more complex types like 'Set<string> | null'
simple_type: simple_type_type_name
           | simple_type '|' simple_type_type_name;
simple_type_type_name: '$' | generic_args | a_type_name ('[]' | '[' a_type_name ']')*?;
a_type_name: ID | SCOPED_NAME;

typescript_type:
                  union_types
                | single_type_spec
                | function_type;

single_type_spec:
                  tuple_type
                | simple_type
                | LeftParen return_type_spec_options_list RightParen '[]'?
                | LeftBrace RightBrace // allow empty maps, {}
                | LeftBrace ~(RightBrace)+? RightBrace // E.g. { light?: T, dark?: T }
                | simple_type_type_name 'is' simple_type;

// [number, string]
tuple_type: '[' tuple_type_args ']';
tuple_type_args: typescript_type
               | tuple_type_args ',' typescript_type;

// I & I2 & I3
// I | I2
// I | I2 & T3
union_types: single_type_spec (('&' | '|') single_type_spec)+;

// () => void
function_type: LeftParen function_definition_params_list? RightParen FAT_ARROW typescript_type;

return_type_spec_options_list: simple_type
                             | return_type_spec_options_list '|' simple_type;

// The type info can be optional, e.g. ?Array<>
//
// Also note that TypeScript allows us to index a tupe, e.g. Parameters<typeof tupleArguments>[0]
generic_type: '<' generic_arg_list '>' index_type_selector*?;
generic_args: '?'? (ID | SCOPED_NAME) generic_type;
index_type_selector: '['  (INTEGER | ID | SCOPED_NAME) ']'; // We can (probably) index via a constant
generic_arg: ~('<' | '>')*? generic_type?;
generic_arg_list: generic_arg
                | generic_arg_list ',' generic_arg;

function_body : function_body_statement*?;

function_body_statement : test_declaration
                        | local_nested_functions // might be a factory function, or a plain wrapper (common in Angular modules)
                        | block_statement
                        | anything;

local_nested_functions: function_declaration;

block_statement : LeftBrace function_body_statement*? RightBrace;

fragment ESCAPED : '\\\\' | '\\"';
LITERAL : '"' ( ESCAPED | ~('\n'|'\r') )*? '"';

fragment ESCAPED_SINGLE : '\\\\' | '\\\'';
SINGLE_LITERAL : '\'' (ESCAPED_SINGLE | ~('\n'|'\r') )*? '\'';

// Note: there's a some nasty traps here:
// 1. Fixed with the ~'>' rule.
//    Without this rule, we would fail to close the block in the following kind of expression:
//       <Flex grow={1}>{systemId && <BadgeEntity systemId={systemId} size="sm" /> } </Flex>
// 2. An expression like </SpanAmount> : ''}<br/> got interpreted as a regex /.../ with a trailing >.
//   This is solved by lexing the HTML close tag separately; cannot be part of a regex without escape.
OBVIOUS_HTML_CLOSE_TAG: '</' [a-zA-Z]+;
fragment ESCAPED_SLASH : '\\/';
REGEX : '/' ~'>' (ESCAPED_SLASH | ~('\n'|'\r') )*? '/';

fragment ESCAPED_BACKTICK : '\\`';
BACKTICK_TEMPLATE : '`' ( ESCAPED_BACKTICK | ~('\n'|'\r') )*? '`';


LITERAL_CHAR : '\'' . '\'' -> skip;

ASSIGN: '=';

NEW: 'new';
ASYNC: 'async';
VAR: 'var';
FUNCTION: 'function' '*'?;
EXPORT: 'export';
CLASS: 'class';

DOLLAR: '$';

// To detect factory functions
RETURN: 'return';

// Test tokens
TEST: 'test';
IT: 'it';
DESCRIBE: 'describe';
BeforeEach: 'beforeEach';
AfterEach: 'afterEach';

FAT_ARROW: '=>';

COLON: ':';
QUESTION_MARK: '?';

SQUARE_LEFT: '[';
SQUARE_RIGHT: ']';

LeftParen : '(';
RightParen: ')';

LeftBrace : '{';
RightBrace : '}';

SEMICOLON : ';';

Whitespace : [ \t]+ -> skip;

BlockComment: '/*' .*? '*/' -> skip;
LineComment: '//' ~[\r\n]* -> skip;

NEWLINE : '\r'? '\n' -> skip;

IF: 'if';
OPERATORS: '&&' | '||' | QUESTION_MARK;
CASE: 'case';
FOR: 'for';
WHILE: 'while';
CATCH: 'catch';
THROW: 'throw';
SWITCH: 'switch';

ID : [a-zA-Z_][a-zA-Z0-9_]*;
SCOPER : '.';
SCOPED_NAME : ID (SCOPER ID)+;

INTEGER: [0-9]+;

ANY_CHAR : .; // Put this lexer rule last to give it the lowest precedence
