grammar CppMicro;

import CppSharedRulesParser;

translationunit
:
	expression* EOF
;

expression : ms_unit_test_suite
           | access_specifiers
           | function_declaration
           | ms_unit_test
           | mc_unit_test_suite
           | mc_unit_test
           | google_unit_test
           | pre_proc_top_level_start
           | using_namespace
           | namespace_block
           | extern_c_block
           | class_declaration
           | block_statement
           | anything;

namespace_block : NAMESPACE (ID | SCOPED_NAME)? LeftBrace expression* RightBrace;

access_specifiers: (PUBLIC | PROTECTED | PRIVATE) COLON;

extern_c_block : EXTERN_C expression* RightBrace;

class_declaration : (CLASS | STRUCT) class_name ~(';' | LeftBrace)*? LeftBrace expression* RightBrace;
class_name : ID | SCOPED_NAME;

// We just want to get rid of the symbol following the #ifdef so that it's not part of a possible subsequent function.
pre_proc_top_level_start : (PRE_PROC_IF | PRE_PROC_IFDEF | PRE_PROC_IFNDEF) pre_proc_top_level_conditions;

// We use a different rule on the top-level since we need to parse more carefully (and significantly slower) in
// order to not consume function signatures that may follow immediately after.
pre_proc_top_level_conditions: pre_proc_top_level_condition
                             | pre_proc_top_level_conditions ('&&' | '||' | '^') pre_proc_top_level_condition;

pre_proc_top_level_condition : '!'?  ('1' | '0' | ID | '=' | DEFINED | (LeftParen ID RightParen))+?
                             | LeftParen pre_proc_top_level_condition RightParen
                             | 'defined' LeftParen pre_proc_top_level_condition RightParen;

// Parsing the ID (e.g. MY_DEFINE) is tricky; we don't want it to be greedy.

google_unit_test : GOOGLE_TEST LeftParen test_case_name ',' test_name RightParen LeftBrace function_body RightBrace;
test_case_name: (ID | SCOPED_NAME |  DEREFERENCED_OBJECT);
test_name: ID;

// Parse Microsoft's CppUnitTest code:
//
ms_unit_test_suite: TEST_CLASS LeftParen ms_unit_test_suite_name RightParen LeftBrace expression* RightBrace;
ms_unit_test_suite_name: ID;
ms_unit_test : ms_type_of_test_method
               LeftParen ms_unit_test_suite_name RightParen LeftBrace function_body RightBrace;
ms_type_of_test_method: (TEST_METHOD | TEST_METHOD_INITIALIZE | TEST_METHOD_CLEANUP | TEST_CLASS_INITIALIZE
                  | TEST_CLASS_CLEANUP | TEST_MODULE_INITIALIZE | TEST_MODULE_CLEANUP);

// Parse MS Minecraft test framework, see https://trello.com/c/cLRclis5/51-support-microsoft-test-framework-for-c-code-health
//
mc_unit_test_suite: BR_TEST_CLASS LeftParen mc_unit_test_suite_name mc_unit_test_args? // Can have varags like  BR_TEST_CLASS(TestName, tag1, tag2)
                    RightParen LeftBrace expression* RightBrace;
mc_unit_test_suite_name: ID;

mc_unit_test_args: ',' mc_unit_test_args_list;
mc_unit_test_args_list : mc_unit_test_arg
                       | mc_unit_test_args_list ',' mc_unit_test_arg;
mc_unit_test_arg: ID | INT | FLOAT | LITERAL | LITERAL_CHAR;

mc_unit_test : mc_type_of_test_method
               LeftParen mc_unit_test_suite_name mc_unit_test_args? // Can have varargs like BR_TEST_METHOD(Check2, aTag, aSecondTag)
               RightParen LeftBrace function_body RightBrace;
mc_type_of_test_method: BR_TEST_METHOD
                      | BR_TEST_CLASS_INITIALIZE
                      | BR_TEST_CLASS_CLEANUP
                      | BR_TEST_METHOD_INITIALIZE
                      | BR_TEST_METHOD_CLEANUP;

function_declaration : 'explicit'? function_name LeftParen function_definition_params_list? RightParen ':' member_init_list throw_spec? LeftBrace function_body RightBrace
                     | 'explicit'? function_name LeftParen function_definition_params_list? RightParen throw_spec? LeftBrace function_body RightBrace

                     // #if (SOME_COMPILE_FLAG)
                     // void foo(x, y, z) {
                     //    foo1();
                     | plain_function_in_preproc_block_if LeftBrace function_body
                     // #else
                     // void foo(a, b, c) {
                     //  foo2();
                        (plain_function_in_preproc_block_else LeftBrace function_body)*?
                     // endif
                        PRE_PROC_ENDIF
                        function_body RightBrace
                     | plain_function_declaration
                     | plain_function_in_preproc_block_if (plain_function_in_preproc_block_else)*? plain_function_in_preproc_block_else PRE_PROC_ENDIF  LeftBrace function_body RightBrace
                     | '~' function_name LeftParen function_definition_params_list? RightParen throw_spec? LeftBrace function_body RightBrace;

// Yes, messy. We need to parse function definitions that might be inside pre-proc declarations. Example:
//   #ifdef MY_DEF_1
//   void foo(int a, int b) {
//   #else
//   static void foo(int a, int b) {
//   #endif
plain_function_in_preproc_block_if : // First rule: #if (COMPILE_FLAG_X == 1) || (COMPILE_FLAG_Y == 1)
                                 ((PRE_PROC_IF | PRE_PROC_IFDEF | PRE_PROC_IFNDEF)
                                    pre_proc_top_level_conditions
                                    plain_function_declaration_start);

plain_function_in_preproc_block_else : // First rule: #elif (COMPILE_FLAG_X == 1) || (COMPILE_FLAG_Y == 1)
                                 (PRE_PROC_ELIF pre_proc_top_level_conditions
                                    plain_function_declaration_start)
                                // Second rule: #else
                                | PRE_PROC_ELSE plain_function_declaration_start;

plain_function_declaration_start: function_type_signature function_name LeftParen function_definition_params_list? RightParen 'const'? throw_spec? trailing_return_type?;

plain_function_declaration: plain_function_declaration_start LeftBrace function_body RightBrace;

// make sure we handle stuff like noexcept(sizeof(T) < 4)
throw_spec : (NOEXCEPT | THROW) ~(LeftBrace | ARROW)*?;

function_type_signature : ~(LeftBrace | RightBrace | LeftParen | RightParen |
                            PRE_PROC_IFDEF | PRE_PROC_IF | PRE_PROC_IFNDEF |
                            PRE_PROC_ENDIF | PRE_PROC_ELSE | SEMICOLON)+?;

member_init_list : member_init_list_param
                 | member_init_list ',' member_init_list_param;
member_init_list_param : member_init_name LeftParen member_init_value? RightParen;
member_init_name : ID | SCOPED_NAME;
member_init_value : (ID | SCOPED_NAME | INT | DEREFERENCED_OBJECT)
                  | (ID | SCOPED_NAME | DEREFERENCED_OBJECT) LeftParen function_param*? RightParen // init with function call
                    (',' member_init_value)*
                  | ~(LeftParen | RightParen)+?;

function_definition_params_list : function_param
                                | function_definition_params_list ',' function_param
                                | function_param+
                                ;

function_param: generic_arg
               | (ID | SCOPED_NAME | INT | DEREFERENCED_OBJECT) LeftParen function_param RightParen // a macro wrapping a parameter
               | ~('=' | LeftBrace)+? '=' ~LeftParen*? LeftParen RightParen // NOTE: note generic enough - could have arguments!
               | ~(LeftParen | RightParen)+?;

// Parse templates as arguments.
// std::function<void(int)>
// std::function<void()>
//
generic_arg: template_type_name '<' template_instantiation*? '>' argument_modifier argument_name;
template_instantiation: template_type_name (LeftParen template_type_name*  RightParen)?
                      | template_type_name '<' template_instantiation '>'; // Type< T<AnotherType> >
template_type_name:  (ID | SCOPED_NAME);
argument_modifier: ('*' | '&')*?;
argument_name: ID;

