grammar BrightScript;

// Grammar for Nested Complexity
//
nested_complexity
:
    nested_complexity_expression* EOF
;

nested_complexity_expression : nested_complexity_element
                             | anything;

nested_complexity_element : for_loop
                          | while_loop
                          | single_line_if
                          | multi_line_if;

for_loop : FOR nested_complexity_expression+? END FOR;
while_loop : WHILE nested_complexity_expression+? END WHILE;

single_line_if: IF ~(THEN | END | ELSEIF | ELSE)+? THEN ~(':' | '?' | ELSEIF | ELSE | END)+? ('?') | ':';

// if BooleanExpression then
//    statements
// elseif something then
//    ...
multi_line_if : IF nested_complexity_expression+? END IF;

// Grammar for Cyclomatic Complexity
//
cyclomatic_complexity
:
	cyclomatic_complexity_expression* EOF
;

cyclomatic_complexity_expression : complexity_terminators
                                 | complexity
                                 | anything;

complexity: loops | paths | conditionals | operators;

// Capture the terminating statements separately, otherwise they'll end
// up being counted as complexity inducing.
complexity_terminators: END FOR
                      | END WHILE
                      | END IF;

loops: FOR | WHILE;
paths: CATCH | GOTO;
operators: AND | OR | ternary_operator;
ternary_operator: Question_Mark; // rough, but nothing else that can match
conditionals: IF | ELSEIF;

// Grammar for identifying function blocks
//
translationunit
:
	expression* EOF
;

expression : class_declaration
           | namespace_declaration
           | function_declaration
           | sub_routine_declaration
           | global_anonymous_function
           | anything;

anything: .;

class_declaration: CLASS class_name expression+ END CLASS;
class_name: ID;

namespace_declaration: NAMESPACE class_name expression+ END NAMESPACE;

function_declaration: function_header function_body function_footer;
function_header: FUNCTION function_name function_arguments (AS function_return_type)?;
function_name: ID;
function_return_type: data_type;
function_footer: END FUNCTION;

// SUB
//
sub_routine_declaration: sub_routine_header sub_body sub_routine_footer;
sub_routine_header:  SUB function_name function_arguments;
sub_routine_footer: END SUB;

// SUB body
//
sub_body: sub_body_statement+;
sub_body_statement: member_data
                  | ~SUB
                  | anonymous_function;

// Function body
//
function_body: function_body_statement+;
function_body_statement: member_data
                      | ~FUNCTION
                      | anonymous_function;

// Member access, needed for cohesion calculations.
// These can be either:
//  - m.someVar
//  - m.top.someGlobalVar
member_data: MEMBER_ACCESS member_data_name;
member_data_name: ID | TOP_COMPONENT;

// Anonymous functions: can be global as well as part of function bodies
//
anonymous_function: anonymous_function_header function_body function_footer;
anonymous_function_header: FUNCTION function_arguments (AS function_return_type)?;

global_anonymous_function: global_anonymous_function_variable_name '=' anonymous_function;
global_anonymous_function_variable_name: ID;

// Function arguments
//
function_arguments: '(' function_definition_params_list? ')';
function_definition_params_list : function_param
                                | function_definition_params_list ',' function_param
                                ;
function_param: function_param_name function_param_default_value? (AS data_type)?;
function_param_name: ID;
data_type: ID;
function_param_default_value: '=' (associative_array | plain_array | .);

// Associative arrays:
//  {myKey: { key1: "value"}}
associative_array: '{' associative_array_content? '}';
associative_array_content: property
                         | associative_array_content ',' property;
property: ID ':' ~(',' | '}')+?
        | ID ':' associative_array
        | ID ':' plain_array;

// Arrays can be complex:
//  node = [ 1, 2, 3 ]
//  nodeType = [ x+5, true, 1<>2, ["a","b"] ]
plain_array: '[' plain_array_content? ']';
plain_array_content: plain_array_member
                   | plain_array_content ',' plain_array_member;
plain_array_member: plain_array
                  | associative_array
                  |  ~(',' | ']')+?;

// Lexer
//
fragment ESCAPED : '\\\\' | '\\"';
LITERAL : '"' ( ESCAPED | ~('\n'|'\r') )*? '"';

// Regex in BrigterScript, eg. print /hello if world/ig
fragment REGEX_DELIM: '/';
REGEX: REGEX_DELIM ~('\n'|'\r')*? REGEX_DELIM;

// BrighterScript string templates which can be multi line:
//  text = `first line if else text
//          second line text`
fragment TEMPLATE_DELIM: '`';
STRING_TEMPLATE: TEMPLATE_DELIM .+? TEMPLATE_DELIM;

// BrightScript is case insensitive
fragment A : [aA];
fragment B : [bB];
fragment C : [cC];
fragment D : [dD];
fragment E : [eE];
fragment F : [fF];
fragment G : [gG];
fragment H : [hH];
fragment I : [iI];
fragment J : [jJ];
fragment K : [kK];
fragment L : [lL];
fragment M : [mM];
fragment N : [nN];
fragment O : [oO];
fragment P : [pP];
fragment Q : [qQ];
fragment R : [rR];
fragment S : [sS];
fragment T : [tT];
fragment U : [uU];
fragment V : [vV];
fragment W : [wW];
fragment X : [xX];
fragment Y : [yY];
fragment Z : [zZ];

Null_Coalescing: '??';
Question_Mark: '?';

LineComment: '\'' ~[\r\n]* -> skip;
REM: R E M;
RemComment: REM ~[\r\n]* -> skip;

MEMBER_ACCESS : M '.';
TOP_COMPONENT : T O P;

FUNCTION: F U N C T I O N;
SUB: S U B;
AS: A S;
END: E N D;
CLASS: C L A S S;
NAMESPACE: N A M E S P A C E;

// Complexity inducing elements
//
FOR: F O R;
IF: I F;
THEN: T H E N;
ELSEIF: E L S E I F;
ELSE: E L S E;
GOTO: G O T O;
WHILE: W H I L E;
CATCH: C A T C H;
AND: A N D;
OR: O R;


ID : [a-zA-Z_][a-zA-Z0-9_\-?]*;
NUMERIC: [0-9]+('.'[0-9])*;

Whitespace : [ \t]+ -> skip;
NEWLINE : '\r'? '\n' -> skip;

ANY_CHAR : .; // Put this lexer rule last to give it the lowest precedence