(ns codescene.features.warnings.warnings-presentation
  (:require
    [clojure.spec.alpha :as s]
    [clojure.java.io :as io]
    [codescene.analysis.analysis-warnings :as analysis-warnings]
    [codescene.analysis.analysis-warning-types :as warning-types]
    [codescene.analysis.paths :as paths]
    [codescene.recommendations.code-health-trends :as code-health-trends]
    [medley.core :refer [update-existing]]
    [taoensso.timbre :as log]
    [codescene.util.json :as json]))

(defn- read-json
  [analysis-results-path json-file]
  (let [file (io/file analysis-results-path json-file)]
    (when (.exists file)
      (json/parse-string (slurp file)))))

(s/def ::category string?)
(s/def ::explanation string?)
(s/def ::description string?)
(s/def ::linkable-name (s/nilable string?))

(s/def ::presentable-warning
  (s/keys :req-un [::description ::linkable-name ::category ::explanation]))

(defn- pluralize
  ([n item] (pluralize n item "s"))
  ([n item ending] (str n " " item (if (= n 1) "" ending))))

;; The author alias merge warning doesn't make sense in cloud since this is already solved.
(defn- warnings-to-ignore
  [warning]
  (let [{warning-type ::analysis-warnings/warning-type} warning]
    (= warning-type ::analysis-warnings/author-alias-merge-suggestions)))

;;; NOTE: when adding new warning types here, it's useful to set up
;;; new nav menu equivalents in `codescene-cloud-web.routes.nav-items/warning->pseudo-path` 

(defmulti analysis-warning->display
  "Formats early warnings for Cloud-specific display in
  templates."
  ::analysis-warnings/warning-type)

(defmethod analysis-warning->display ::analysis-warnings/branch-long-ttl
  [analysis-warning]
  (let [warning-count (+ (::analysis-warnings/yellow-warnings analysis-warning 0)
                         (::analysis-warnings/red-warnings analysis-warning 0))]
    (merge warning-types/branch-ttl-warning
           {:description (pluralize warning-count "Long Lived Branch" "es")
            :linkable-name "branch-long-ttl"
            :domain "delivery"})))

;;; The "rising hotspots" warning might be deprecated. If so, we could remove this method.
;;; NB: some of the supporting rising-hotspots code is still required for
;;; the `rising-code-health-issues-from` warning. 
(defmethod analysis-warning->display ::analysis-warnings/rising-hotspots
  [analysis-warning]
  (let [hotspot-count (count (::analysis-warnings/hotspots analysis-warning))]
    (merge warning-types/rising-spot-warning
           {:description (pluralize hotspot-count "Rising Hotspot")
            :linkable-name "rising-hotspots"
            :domain "code-health"})))

(defmethod analysis-warning->display ::analysis-warnings/complexity-increase
  [{:keys [::analysis-warnings/yellow-markers
           ::analysis-warnings/red-markers]}]
  (let [n-warnings (+ red-markers yellow-markers)]
    (merge warning-types/complexity-trend-warning
           {:description (pluralize n-warnings "Complexity Warning")
            :linkable-name "complexity-increase"
            :domain "code-health"})))

(defmethod analysis-warning->display ::analysis-warnings/ex-authors
  [{:keys [::analysis-warnings/authors]}]
  (let [n-warnings (count authors)]
    (merge warning-types/exdevs-trend-warning
           {:description (pluralize n-warnings "Possible Ex-Developer")
            :linkable-name "ex-authors"
            :domain "knowledge-distribution"})))

(defmethod analysis-warning->display ::analysis-warnings/teamless-authors
  [{:keys [::analysis-warnings/authors]}]
  (let [n-warnings (count authors)]
    (merge warning-types/teamless-authors-warning
           {:description (pluralize n-warnings "Teamless Author")
            :linkable-name "teamless-authors"
            :domain "team-code-alignment"})))

(defmethod analysis-warning->display ::analysis-warnings/high-risk-commits
  [analysis-warning]
  (let [{number-of-risk-commits ::analysis-warnings/number-of-risk-commits} analysis-warning]
    (merge warning-types/high-risk-commits-warning
           {:description (pluralize number-of-risk-commits "High Risk Commit")
            :linkable-name "high-risk-commits"
            :domain "code-health"})))

(defmethod analysis-warning->display ::analysis-warnings/author-alias-merge-suggestions
  [{:keys [::analysis-warnings/suggestion-count]}]
  (merge warning-types/author-alias-merge-suggestions-warning
         {:description (str (pluralize suggestion-count "author")
                            " can be automatically merged.")
          :linkable-name "author-alias-merge"
          :domain "knowledge-distribution"}))

(defmethod analysis-warning->display ::analysis-warnings/notes-category-warning
  [{:keys [::analysis-warnings/supervision-warnings]}]
  (merge warning-types/augmented-analysis-warning
         {:description (pluralize supervision-warnings "Missed goal")
          :linkable-name "supervised-spot"
          :domain "code-health"}))

(defmethod analysis-warning->display :default
  [{:keys [::analysis-warnings/warning-type]}]
  (if (keyword? warning-type)
    (log/warnf "Unhandled warning type: %s" warning-type)
    (log/warnf "Missing warning type in early warning"))
  {:prio 0
   :rank 100
   :name "Unrecognized warning"
   :description (if (keyword? warning-type)
                  (format "Error: unknown warning type. %s" warning-type)
                  "Error: incorrect early warning.")
   :category "danger"
   :explanation "Mismatch between analysis versions. Please report this."
   :linkable-name nil})

(defn- code-health-degradation-warning-from
  [{:keys [monthly yearly] :as _code-health-recommendations}]
  (let [n-month (count monthly)
        n-year (count yearly)
        n-degrading (+ n-month n-year)]
    (when (> n-degrading 0)
      (merge warning-types/code-health-decline-warning
             {:description (if (= n-degrading 1)
                             "1 file declines in Code Health"
                             (str n-degrading " files decline in Code Health"))
              :linkable-name "code-health-degradation"
              :domain "code-health"}))))

(defn- predict-code-health-degradation-from
  [{:keys [predictions] :as _code-health-recommendations}]
  (let [n-degrading (count predictions)]
    (when (> n-degrading 0)
      (merge warning-types/code-health-predicted-decline-warning
             {:description (if (= n-degrading 1)
                             "Prediction: 1 file is likely to decline in Code Health"
                             (str "Prediction: " n-degrading " files are likely to decline in Code Health"))
              :linkable-name "predicted-code-health-degradation"
              :domain "code-health"}))))

(defn- rising-code-health-issues-from
  [rising-spots]
  (let [n-degrading (count rising-spots)]
    (when (> n-degrading 0)
      (merge warning-types/rising-hotspots-code-health-issues
             {:description (str (if (= n-degrading 1)
                                  "1 file climbs on the hotspot ranking, and that file has declining Code Health."
                                  (str n-degrading " files are climbing on the hotspot ranking, and have declining Code Health."))
                                " Investigate and Supervise to ensure it isn't a growing concern.")
              :linkable-name "rising-hotspots-code-health-issues"
              :domain "code-health"}))))

(defn- code-health-improvements-from
  [analysis-results-path {:keys [improvements] :as _code-health-recommendations}]
  ; a new type of analysis: don't present the info unless it exists (the summary was generated before...):
  (when (code-health-trends/has-improvement-scores? (partial paths/make-analysis-path-to analysis-results-path))
    (let [n-improved (count improvements)]
      (when (> n-improved 0)
        (merge warning-types/code-health-improvement
               {:description (if (= n-improved 1)
                               "1 file improves in Code Health"
                               (str n-improved " files improve in Code Health"))
                :linkable-name "code-health-improved"
                :domain "code-health"})))))

(defn- hotspots-code-health-evolution
  [analysis-results-path]
  (if-let [p (read-json analysis-results-path paths/code-health-trend-recommendations-json)]
    (let [code-health-evolution (:code-health-degradations p)]
      [(code-health-degradation-warning-from code-health-evolution)
       (predict-code-health-degradation-from code-health-evolution)
       (code-health-improvements-from analysis-results-path code-health-evolution)])
    []))

(defn- rising-hotspots-code-health-warnings
  [analysis-results-path]
  (if-let [p (:markers (read-json analysis-results-path paths/rising-hotspots-code-health-issues-json))]
    [(rising-code-health-issues-from p)]
    []))

(defn- code-health-warnings
  [analysis-results-path]
  (into []
        (remove nil?)
        (concat
          (hotspots-code-health-evolution analysis-results-path)
          (rising-hotspots-code-health-warnings analysis-results-path))))

(s/fdef get-analysis-early-warnings
        :ret :codescene.analysis.specs/analysis-warnings)
(defn get-analysis-early-warnings
  [analysis-results-path]
  (->> (read-json analysis-results-path paths/early-warnings-json)
       (map #(update-existing % ::analysis-warnings/warning-type keyword))))

(defn- fix-a-backwards-incompatible-ui
  "For some reason, the UI gets confused when both prio and category is present and uses
   prio -- this means improvements get classified as danger:(
   Fix that here."
  [ws]
  (map (fn [w] (dissoc w :prio)) ws))

(s/fdef get-supported-early-warnings
        :ret (s/coll-of ::presentable-warning))
(defn get-supported-early-warnings
  [analysis-results-path]
  (->> (get-analysis-early-warnings analysis-results-path)
       (remove warnings-to-ignore)
       (map analysis-warning->display)
       (concat (code-health-warnings analysis-results-path))
       (sort-by :prio)
       fix-a-backwards-incompatible-ui))


