(ns codescene.features.util.retry
  (:require [taoensso.timbre :as log]))

(defn exponential-backoff
  "This will retry after waiting 'time' ms the first time, then '* rate' it
  each time. When it waits 'max' ms, it won't retry any more. Predicate 'p?'
  applied on the Throwable decides whether to retry or fail."
  [{:keys [time rate max p?]
    :or {time 250 rate 2 max 2000 p? (constantly true)}
    :as opts} f]
  (try
    (f)
    (catch Throwable t
      (cond
        (>= time max) (do (log/infof "exponential-backoff: max time reached will not retry" t)
                          (throw t))
        (p? t) (do
                 (log/infof t "exponential-backoff: caught exception. Will retry in %s ms" time)
                 (Thread/sleep (long time))
                 (exponential-backoff (assoc opts :time (* time rate)) f))
        :else (do
                ;; log only exception message, not full stacktrace
                ;; (annoying when we don't want to retry (such as 404 on S3 download))
                (log/info "exponential-backoff: non-handled exception, will not retry." t)
                (throw t))))))

(defmacro try-backoff
  "See `exponential-backoff` for options description.
  Use like:
   (try-backoff {}
     (println \"trying!\")
     (do-some-stuff))"
  [opts & body]
  `(exponential-backoff ~opts (fn [] ~@body)))
