(ns codescene.features.util.number
  "Handy utility functions for dealing with numbers."
  (:require [codescene.presentation.display :as display]))

(defn floor-with-precision
  "Round down a double to the given precision (number of significant digits)"
  [precision d]
  (let [factor (Math/pow 10 precision)]
    (/ (Math/floor (* d factor)) factor)))

(defn round-with-precision
  "Round double to the given precision (number of significant digits)"
  [precision d]
  (let [factor (Math/pow 10 precision)]
    (/ (Math/round (* d factor)) factor)))

(defn floor-number-when-exists
  "round down a double with 2 decimals or return '-' if not coerceable to double"
  [d]
  (let [value (display/->maybe-double d)]
    (if (number? value) (floor-with-precision 2 value) "-")))

(def high-precision-cutoff 1.1)                             ; the cut-off for where high-precision is needed

(defn round-code-health-scores
  "We're now using a code health curve with exponential decay. That means, the closer we
   get to 1.0, the more precision we include.
   To make sure the code health scores are user-friendly, we round everything above 1.1. Once
   we get lower scores, we instead use the full precision so that even minor declines/improvements
   become visble."
  [code-health-score]
  (if (< code-health-score high-precision-cutoff)
    code-health-score
    (floor-with-precision 2 code-health-score)))

(defn as-int [x]
  (if (integer? x)
    x
    (Integer/parseInt x)))
