(ns codescene.features.system-map-filtering
  (:require
   [codescene.analysis.paths :as paths]
   [clojure.java.io :as io]
   [codescene.features.util.csv :as csv]
   [clojure.string :as string])
  (:import
   (java.time LocalDateTime)
   (java.time.format DateTimeFormatter)))


(def issues-per-automplete-request 10)

;;; TODO @josf: This is probably not the right place to put the autocomplete logic
;;; Most of this should go to features
(defn- offset-time-string-as-instant
  [offset-date-time]
  (try
    (LocalDateTime/parse offset-date-time DateTimeFormatter/ISO_OFFSET_DATE_TIME)
    (catch Exception e
      nil)))

(defn- date-comparator-desc
  "We handle nils here so that any issues without a latest for some
  reason will end up at the bottom of the list."
  [d1 d2]
  (cond
    (and d1 d2)
    (.compareTo d2 d1)
    (every? not [d1 d2]) 0
    d1 1
    d2 -1))

(defn- sort-dates-desc [ds]
  (sort date-comparator-desc  ds))

(comment
  (sort-dates-desc
   [(offset-time-string-as-instant "2023-06-11T16:42:46Z")
    (offset-time-string-as-instant "2021-11-23T10:17:38Z")]))

(defn- id-matches-query?
  [query id]
  (string/includes? id query))

(defn- normalize [query-or-id]
  (let [normalized-for-length (if (>  (count query-or-id) 40)
                                (subs query-or-id 0 40)
                                query-or-id)]
    (-> normalized-for-length
        string/upper-case
        (string/replace  #"\s" ""))))

(defn- issue-with-latest-date
  "Appends a `:latest` field that correponds to the latest of all the date fields."
  [{:keys [last-commit closed first-commit created] :as issue}]
  (let [latest (->> [last-commit closed first-commit created]
                    (keep offset-time-string-as-instant)
                    sort-dates-desc
                    first)]
    (assoc issue :latest latest)))

(defn autocomplete-pm-ids-by-query
  [issues query]
  (let [normalized-query (normalize query)
        id-matcher (if (not-empty query)
                     (fn [{:keys [id]}] (id-matches-query? normalized-query (normalize id)))
                     (constantly true))]
    (->> issues
         (filter id-matcher)
         (map issue-with-latest-date)
         (sort-by :latest date-comparator-desc)
         (map :id))))

;;; TODO @josf: this is fairly slow (500ms on my laptop with a 20K LoC evo file)
(defn matching-files-from
  [issue-set log-file-items]
  (into #{}
        (comp
          (filter #(issue-set (:ticket-id %)))
          (map :entity))
        log-file-items))

(comment
  (time (matching-files-from #{"CODESCENE-1838"} (csv/read-csv  (io/file "/Users/joseph/Empear/testing/analyses-3/codescene/analysis20241015072125/pm_evo_data.csv")))))

(defn files-for-ids
  [pm-log-file issue-ids]
  (if (not pm-log-file)
    {:files [] :status "unavailable" :issues issue-ids}
    (let [issue-set (set issue-ids)]
      {:status "ok"
       :issues issue-ids
       :files (->> pm-log-file
                   csv/read-csv 
                   (matching-files-from issue-set)
                   (into []))})))

;;; Validator for use in the handler
(defn invalid-body-json?
  [body-params]
  (cond
    (not (sequential? body-params)) "Must be an array"
    (not (every? string? body-params)) "All items must be strings"
    :else nil))


(comment
  (sort-by
    :latest
    date-comparator-desc
    (map
      (fn [{:keys [last-commit closed first-commit created] :as issue}]
        (let [latest (->>
                       [last-commit closed first-commit created]
                       (keep offset-time-string-as-instant)
                       sort-dates-desc
                       first)]
          (assoc issue :latest latest)))
      (csv/read-csv
        (io/file  (:dir  (analysis-feature/get-analysis-by-id (db/persistent-connection) 33 1966))
                  paths/pm-issues-csv)))))

