(ns codescene.features.stripe.webhook
  "Contains a generic function for handling Stripe webhooks.

  This function will check the signature, determine the stripe account this webhook
  is coming from (and add that to payload) and run the handler function.

  It automatically handled ping event and invalid payloads."
  (:require [buddy.core.codecs :as codecs]
            [buddy.core.mac :as mac]
            [clojure.spec.alpha :as s]
            [clojure.string :as string]
            [codescene.cache.core :as cache]
            [codescene.util.http :as http]
            [codescene.util.json :as json]
            [ring.util.response :as response]
            [slingshot.slingshot :refer [try+ throw+]]
            [taoensso.timbre :as log])
  (:import (java.nio.charset StandardCharsets)
           (org.apache.commons.codec DecoderException)))

(defn- calculate-signature
  [secret payload]
  (-> (mac/hash payload {:key secret :alg :hmac+sha256})
      (codecs/bytes->hex)))

(defn- valid-signature?
  [secret payload signature timestamp]
  (let [content (str timestamp "." payload)]
    (if (mac/verify content
                    (codecs/hex->bytes signature)
                    {:key secret :alg :hmac+sha256})
      true
      (log/debugf "Signature mismatch, theirs: %s, ours: %s" signature (calculate-signature secret content)))))

(defn verify-signature
  "Checks whether payload has a valid signature
  and returns corresponding stripe account, either `:ab` or `:inc`."
  [payload signature timestamp account-map]
  (some (fn [[stripe-account secret]]
          (when (valid-signature? secret payload signature timestamp)
            stripe-account))
        account-map))

(defn response [status message]
  (-> (http/response message status)
      (response/header "cache-control" "no-store, private")
      (response/charset (.name StandardCharsets/UTF_8))))

;; used as switching function if the below defmulti
(defn extract-event-type
  "Extract the event type from the payload. Useful for multi-method switching."
  [payload]
  (let [formatter (fn [s] (some-> s
                                  string/lower-case
                                  keyword))
        event-type (formatter (:type payload))]
    (log/debug "Switching on:" event-type)
    event-type))

(defn split-signature-header [header-name header-data]
  (let [elements (string/split (or header-data "") #",")]
    (when (> 2 (count elements))
      (throw+ {:type    :invalid-signature
               :message (format "Invalid signature in header %s, expecting at least two elements: '%s'" header-name header-data)}))

    (let [pairs (->> elements
                     (mapcat #(string/split % #"=")))]
      (when (odd? (count pairs))
        (throw+ {:type    :invalid-signature
                 :message (format "Invalid signature in header %s, expecting pairs key=val: '%s'" header-name header-data)}))

      (-> (apply hash-map pairs)
          (select-keys ["t" "v1"])
          (update "t" parse-long)))))

(defn handle-payload
  {cache/conf (cache/size< 1000 (fn [payload _] (-> payload :request :idempotency_key)))}
  ;; make sure we don't handle same event multiple times
  [payload handler]
  (case (:type payload)
    nil (response 500 (doto "No event type found in Stripe webhook payload" log/error))
    :ping (let [{:keys [type id]} payload]
            (response 200 (format "%s received with id %s" type id)))
    (or (handler payload)
        (response 500 (doto (format "Not yet handling Stripe webhook event type '%s'" (:type payload))
                             log/info)))))

(cache/memo #'handle-payload)

(defn clear-idempotency-cache
  "Useful for tests so multiple attempts with same payload will still call the underlying."
  []
  (cache/memo-clear! handle-payload))

(defn validate-max-age [timestamp max-age]
  (let [ts-now (long (/ (System/currentTimeMillis) 1000.))
        ts-diff (- ts-now timestamp)]
    (when-not (>= max-age ts-diff)
      (throw (ex-info (format "Signature time stamp %s is too old: %s, max allowed age: %s" timestamp ts-diff max-age)
                      {:type :webhook-message-ignored})))))

(s/fdef handle-stripe-webhook
        :args (s/cat :req map?
                     :handler ifn?
                     :accounts (s/map-of keyword? string?)
                     :max-age pos-int?))

(defn handle-stripe-webhook
  "Handles stripe webhook, performing the signature validation step and delegating to process-webhook-event.

  process-webhook-event function doesn't need to handle :ping messages and nil event type, as these will be handled
  by this function.

  Account map is account -> stripe secret map"
  [{:keys [body headers]} process-webhook-event account-map max-age]
  (try+
    (let [stripe-signature-header "stripe-signature"
          _ (log/debugf "[Headers] %s: %s" stripe-signature-header (get headers stripe-signature-header))

          header-data (get headers stripe-signature-header)
          signature-map (split-signature-header stripe-signature-header header-data)
          signature (get signature-map "v1")
          timestamp (get signature-map "t")
          json-payload (slurp body)]
      (validate-max-age timestamp max-age)
      (try+
        (if-let [stripe-account (verify-signature json-payload signature timestamp account-map)]
          (let [payload (json/parse-string json-payload)
                {:keys [type id created api_version]} payload]
            (log/infof "Received Stripe %s webhook with id %s, created at %s; API version: \"%s\"" type id created api_version)
            (log/trace (with-out-str (clojure.pprint/pprint payload)))
            (handle-payload (assoc payload :stripe-account stripe-account
                                           :type (extract-event-type payload))
                            process-webhook-event))
          (response 403 "Signatures didn't match!"))
        (catch DecoderException _
          (log/errorf (:throwable &throw-context) "Problem decoding signature hash: %s" signature)
          (response 403 "Problem decoding signature"))))
    (catch [:type :webhook-message-ignored] _
      ;; something in the data made it impossible to process the message (unknown customer, price, ...)
      ;; we don't want this to be re-sent and too many failures cause Stripe to disable webhook, pending
      ;; manual reenabling
      (log/error (:throwable &throw-context) (:message &throw-context))
      (response 200 "Message ignored"))
    (catch Object _
      (log/error (:throwable &throw-context) (:message &throw-context))
      (response 500 "An unhandled error occurred"))))