(ns codescene.features.security.session
  (:require [clj-time.coerce :as tcc]
            [clj-time.core :as tc]
            [crypto.random :as random]
            [ring.middleware.session.store :as store]))

(defn expired? [session max-age]
  (<= (+ (or (:session/ts session) 0)
         (or (:session/max-age session) max-age))
      (tcc/to-epoch (tc/now))))

(defn new-anti-forgery-token-if-needed
  [session]
  (if (:ring.middleware.anti-forgery/anti-forgery-token session)
    (assoc session :ring.middleware.anti-forgery/anti-forgery-token
                   (random/base64 60))
    session))

(defn recreate-session
  "Mark session to be recreated. Makes sure that some of the other properties are reset too."
  [session]
  (-> session
      (new-anti-forgery-token-if-needed)
      ;; we need to reset session start timestamp
      (dissoc :session/ts)
      (vary-meta assoc :recreate true)))

(defrecord ExpiryCheckingStore
  [delegate session-max-age]
  store/SessionStore
  (read-session [store key]
    (let [session (store/read-session delegate key)]
      (if (expired? session session-max-age)
        (do (store/delete-session store key) nil)
        session)))
  (write-session [store key data]
    (store/write-session delegate key (update data :session/ts #(or % (tcc/to-epoch (tc/now))))))
  (delete-session [store key]
    (store/delete-session delegate key)))

(defn limit-session-age
  "Wraps session store in settings with one that will prevent session older than given
  max from being used."
  ([middleware-defaults store max-age-sec]
   (-> middleware-defaults
       (assoc-in [:session :store] store)
       (limit-session-age max-age-sec)))
  ([middleware-defaults max-age-sec]
   (-> middleware-defaults
       (update-in [:session :store] #(->ExpiryCheckingStore % max-age-sec))
       ;; cookie that holds the session identifier (or the session as a whole)
       ;; doesn't need to outlast the max-age under any circumstance
       (assoc-in [:session :cookie-attrs :max-age] max-age-sec))))
