(ns codescene.features.repository-provider.specs
  (:require [clojure.spec.alpha :as s]
            [codescene.features.client.api :as api]
            [codescene.specs :as specs]))

;; Nillable because in some cases, e.g. when creating a repo-info map from clone URL
;; external ID is not available. In general the application is made to work without it.
(s/def ::external-id (s/nilable ::specs/non-empty-string))
(s/def ::provider-id ::specs/provider-id)
(s/def ::owner-login ::specs/non-empty-string)
(s/def ::repo-slug ::specs/non-empty-string)
(s/def ::sha ::specs/short-commit-hash)

(s/def ::repo-info
  (s/keys :req-un [::provider-id
                   ::owner-login
                   ::repo-slug]
          :opt-un [::external-id]))

(s/def ::commit-info
  (s/keys :req-un [::provider-id
                   ::owner-login
                   ::repo-slug
                   ::external-id
                   ::sha]))

(s/def ::clone-url ::specs/non-empty-string)
(s/def ::url ::clone-url)
(s/def ::auth-token #(satisfies? api/ApiClient %))

(defn repo=
  "Returns true if these denote same repo, the complexity is because some Cloud GitHub DB entries don't
  have external ID and the difference between name and repo-slug."
  [{p1 :provider-id e1 :external-id :as repo1}
   {p2 :provider-id e2 :external-id :as repo2}]
  (and (= p1 p2)
       (if (or (nil? e1) (nil? e2))
         (and (= (:owner-login repo1) (:owner-login repo2))
              (= (:name repo1 (:repo-slug repo1)) (:name repo2 (:repo-slug repo2))))
         (= e1 e2))))
