(ns codescene.features.repository-provider.providers
  (:require [clojure.spec.alpha :as s]
            [codescene.features.repository-provider.specs :as specs]
            [taoensso.timbre :as log]))

;; Special "branch name" used when no branches can be loaded for a repository
;; See `get-branches`
(def no-branch-fallback "<NO BRANCH INFO>")

(def no-branch-results {:default-branch no-branch-fallback
                        :refs (sorted-set no-branch-fallback)
                        :no-branch-fallback? true})

(defn branch-ref->branch-name [branch-ref]
  (or (->> branch-ref (re-matches #"refs/heads/(.*)") second)
      branch-ref))

(defn ->branch-results
  "Performs cleanup of paging state into final shape for repo branch returns."
  [{:keys [items default-branch no-branch-fallback?]}]
  {:refs (into (sorted-set) items)
   :default-branch default-branch
   :no-branch-fallback? (or no-branch-fallback? false)})

(defn branch-defaults-fn
  "Returns a function that loads and adds the default branch to repo info for the repository
  or else fallback value."
  [authed-client default-branch-fn]
  (fn [repo-info]
    (merge repo-info
           (try
             {:default-branch (or (default-branch-fn authed-client repo-info) "master")
              :no-branch-fallback? false}
             (catch Exception e
               (log/error e)
               no-branch-results)))))

(s/fdef commit-info
        :args (s/cat :repo-info ::specs/repo-info
                     :sha ::specs/sha)
        :ret ::specs/commit-info)
(defn commit-info [repo-info sha] (assoc repo-info :sha sha))

(s/fdef repo-info->html-url
        :args (s/cat :repo-info ::specs/repo-info)
        :ret ::specs/url)
(defn repo-info->html-url
  [{:keys [host provider-id owner-login repo-slug]}]
  (case provider-id
    :azure (format "https://%s/%s/_git/%s" (or host "dev.azure.com") owner-login repo-slug)
    :github (format "https://%s/%s/%s" (or host "github.com") owner-login repo-slug)
    :bitbucket (format "https://%s/%s/%s" (or host "bitbucket.org") owner-login repo-slug)
    :gitlab (format "https://%s/%s/%s" (or host "gitlab.com") owner-login repo-slug)
    nil))

(s/fdef repo-info->commit-url
        :args (s/cat :repo-info ::specs/repo-info
                     :commit-sha ::specs/sha)
        :ret ::specs/url)
(defn repo-info->commit-url
  [{:keys [provider-id] :as repo-info} sha]
  (some-> repo-info
          repo-info->html-url
          (str (case provider-id
                 :azure "/commit/"
                 :bitbucket "/commits/"
                 :github "/commit/"
                 :gitlab "/-/commit/")
               sha)))

(s/fdef repo-info->branch-url
        :args (s/cat :repo-info ::specs/repo-info
                     :branch string?)
        :ret ::specs/url)
(defn repo-info->branch-url
  [{:keys [provider-id] :as repo-info} branch-name]
  (some-> repo-info repo-info->html-url
          (str (case provider-id
                 :azure (str "?path=%2F&version=GB")
                 :bitbucket (str "/branch/")
                 :github (str "/tree/")
                 :gitlab (str "/-/tree/"))
               branch-name)))

(s/fdef repo-info->pr-url
        :args (s/cat :repo-info ::specs/repo-info
                     :pr-id (s/nilable string?))
        :ret ::specs/url)
(defn repo-info->pr-url
  [{:keys [provider-id] :as repo-info} pr-id]
  (some-> (and pr-id repo-info)
          repo-info->html-url
          (str (case provider-id
                 :azure "/pullrequest/"
                 :bitbucket "/pull-requests/"
                 :github "/pull/"
                 :gitlab "/-/merge_requests/")
               pr-id)))

(defn provider-name [provider-id]
  (get {:github "GitHub" :gitlab "GitLab" :azure "Azure DevOps" :bitbucket "Bitbucket"} provider-id))

(defn negative-skip?
  "Returns true if the reason for skipped analysis is a negative one, which should be reported to user."
  [reason]
  (#{:delta-analysis/account-trial-expired
     :delta-analysis/account-blocked
     :delta-analysis/multiple-enabled-projects
     :git-branch-head-moved
     :max-commits-limit
     :max-files-limit} (:type (ex-data reason))))
