(ns codescene.features.repository-provider.plugin-gerrit.runs
  (:require [codescene.features.client.api :as api]
            [codescene.features.delta.file-regions :as file-regions]
            [codescene.features.util.template :as template]
            [codescene.features.delta.result.pull-request :as result]
            [codescene.features.repository-provider.plugin-gerrit.delta :as g.delta]))

(defn finding->result [{:keys [description change-level title-prefix category file-name category-description location
                               finding-url suppress-url] :as ctx}
                       markdown?
                       negative?
                       {:keys [patch pr-html-url]}]
  (let [{:keys [start end after?]} location
        title (str title-prefix " " category)
        warning-status (if negative? "ERROR" "WARNING")
        gerrit-url (str (api/to-url pr-html-url patch file-name) "#" start)
        ctx (assoc ctx :gerrit-url gerrit-url)]
    {:category (case change-level
                 :warning warning-status
                 :notice "INFO"
                 :improvement "SUCCESS"
                 nil)
     :summary title
     :message (if markdown?
                (g.delta/strip-html (template/render-file "templates/delta/gerrit-file-comment.md" ctx))
                (cond-> (str title "\n" description)
                  category-description (str ". " category-description)))
     :links [{:url finding-url :primary true :icon "external"}]
     :externalId suppress-url
     :codePointers [{:path (str patch "/" file-name)
                     :range {:start_line start
                             :start_character 0
                             :end_character 0
                             :end_line end}}]}))

(defn results [pr-presentable locator gerrit-location]
  (->> (result/file-comment-items pr-presentable locator)
       (mapv #(finding->result % true (result/negative-review? pr-presentable) gerrit-location))))

(defn main-result [pr-presentable markdown?]
  {:category (if (result/negative-review? pr-presentable) "ERROR" "SUCCESS")
   :summary (str "Check Result: " (if (result/negative-review? pr-presentable) "FAILED" "OK"))
   :message (if markdown?
              (result/presentable->comment pr-presentable :gerrit false)
              (result/presentable->text-comment pr-presentable false))})

(defn check-run [pr-presentable gerrit-location]
  (let [locator (result/cached-locator #(comp first (file-regions/change-locator % false)))]
    {:checkName "CodeScene"
     :checkLink (:result-url pr-presentable)
     :status "COMPLETED"
     :statusDescription (str "Check Result: " (if (result/negative-review? pr-presentable) "FAILED" "OK"))
     ;; statusLink
     :results (cons (main-result pr-presentable true) (results pr-presentable locator gerrit-location))}))

(defn fetch-response [pr-presentable gerrit-location]
  (let [pr-presentable (g.delta/add-default-gates pr-presentable)]
    {:responseCode "OK"
     :links [{:url (:result-url pr-presentable) :primary true :linkIcon "external" :tooltip "See results in CodeScene"}]
     :runs [(check-run pr-presentable gerrit-location)]}))

(defn skipped-response [{:keys [title desc]}]
  {:responseCode "OK"
   :runs [{:checkName "CodeScene"
           :status "COMPLETED"
           :statusDescription "Skipped"
           ;; statusLink
           :results [{:category "INFO"
                      :summary "Check Result: Skipped"
                      :message (format "### CodeScene PR Check Skipped\n%s\n---\n%s" title desc)
                      :externalId "RERUN"}]}]})

(defn error-response [{:keys [title desc]}]
  {:responseCode "OK"
   :runs [{:checkName "CodeScene"
           :status "COMPLETED"
           :statusDescription "Error"
           ;; statusLink
           :results [{:category "ERROR"
                      :summary "Check Result: Error"
                      :message (format "### CodeScene PR Check\n%s\n---\n%s" title desc)
                      :externalId "RERUN"}]}]})

(defn error-delta [{:keys [result]}]
  (if (= :error (:error result))
    (error-response result)
    (skipped-response result)))

(defn ongoing-delta [params]
  (when (g.delta/ongoing-delta? (select-keys params [:repo :review-id :change-id :sha]))
    {:responseCode "OK"
     :runs [{:checkName "CodeScene"
             :status "RUNNING"
             :statusDescription "Check Result: Running"
             ;; statusLink
             :results []}]}))