(ns codescene.features.repository-provider.plugin-gerrit.delta
  (:require [clojure.string :as str]
            [codescene.cache.core :as cache]
            [codescene.delta.detectors.quality-gates :as qg]
            [codescene.features.delta.protocols :as protocols]
            [codescene.features.delta.result.pull-request :as result]
            [codescene.features.repository-provider.providers :as providers]
            [memento.core :as memento]
            [taoensso.timbre :as log]))

(defn ongoing-delta?
  {cache/conf (cache/ttl [1 :h] #(select-keys % [:review-id :change-id :sha]))}
  [params] nil)

(cache/memo #'ongoing-delta?)

(defn change-ongoing-delta
  "This mechanism is because OnPrem doesn't keep pending analyses"
  [full-changes-id revision-id review-id status]
  (let [k {:review-id review-id
           :change-id full-changes-id
           :sha revision-id}]
    (if status
      (memento/memo-add! ongoing-delta? {(list k) status})
      (memento/memo-clear! ongoing-delta? k))
    nil))

(defn strip-html
  "Naive HTML stripping, sufficient to remove html tags we put in our comments, because
  Gerrit markdown doesn't support HTML tags."
  [s] (str/replace s #"<[a-zA-Z/]+>" ""))

(defn save-delta-skipped [insert-error provider-ref reason]
  (log/debugf "provider=plugin-gerrit; action=delta-analysis-skipped; ref=%s" provider-ref)
  (when (or (-> provider-ref :result-options :always-comment?) (providers/negative-skip? reason))
    (insert-error
      {:title (result/render-error-title reason)
       :desc (result/render-error reason)
       :error :skip})))

(defn save-delta-error [insert-error provider-ref error]
  (log/debugf "provider=plugin-gerrit; action=delta-analysis-error; ref=%s" provider-ref)
  (insert-error {:title (result/render-error-title error)
                 :desc (result/render-error error)
                 :error :error}))

(defn add-default-gates
  "Add medium profile to all file results if missing. This is for legacy PRs."
  [pr-presentable]
  (let [add (fn [fr] (update fr :enabled-gates #(or % (qg/profile "medium" {}))))]
    (update-in pr-presentable [:result :file-results] #(mapv add %))))

(defrecord PluginGerritDelta [insert-error]
  protocols/DeltaResultBoundary
  (delta-pending [this provider-ref _result-url]
    (log/debugf "provider=plugin-gerrit; action=delta-analysis-pending; ref=%s" provider-ref))
  (delta-results [this provider-ref _pr-presentable]
    (log/debugf "provider=plugin-gerrit; action=delta-analysis-results; ref=%s" provider-ref))
  (delta-skipped [this provider-ref reason result-url]
    (log/debugf "provider=plugin-gerrit; action=delta-analysis-skipped; ref=%s" provider-ref)
    (save-delta-skipped insert-error reason result-url))
  (delta-error [this provider-ref error _result-url]
    (log/debugf "provider=plugin-gerrit; action=delta-analysis-error; ref=%s" provider-ref)
    (save-delta-error insert-error provider-ref error)))