(ns codescene.features.repository-provider.github.review
  (:require [clojure.set :as set]
            [clojure.string :as str]
            [codescene.features.delta.result.pull-request :as result]
            [codescene.features.client.api :refer [to-url]]
            [codescene.features.repository-provider.github.api :as api]
            [codescene.features.util.template :as template]
            [medley.core :refer [update-existing]]
            [taoensso.timbre :as log]
            [slingshot.slingshot :refer [try+]]))

(defn run-tag [check-name]
  (format "\n[//]: # (%s)" check-name))

(defn add-run-tag [body check-name]
  (str (run-tag check-name) "\n" body))

(defn default-from-coll
  "If key k is missing from an item in coll, then a value is taken from coll2 to set that value"
  [coll k coll2]
  (:ret
    (reduce (fn [{:keys [defaults ret] :as acc} item]
              (if (some? (item k))
                (update acc :ret conj item)
                {:defaults (rest defaults)
                 :ret (conj ret (assoc item k (first defaults)))}))
            {:defaults coll2
             :ret []}
            coll)))

(defn rest-api->internal [comment]
  {:comment/link (:html_url comment)
   :comment/node-id (:node_id comment)
   :comment/raw (-> comment
                    (set/rename-keys {:start_side :startSide :start_line :startLine})
                    (select-keys api/review-comment-keys)
                    (update-existing :startSide #(some-> % symbol))
                    (update-existing :side #(some-> % symbol)))})

(defn create-review!
  "Creates a pending review, prepare data structure representing the review"
  [{:review/keys [pull-request-url sha] :as review} authed-client]
  (log/debugf "Creating review at %s sha=%s" pull-request-url sha)
  (try+
    (let [resp (api/api-request authed-client
                                :post
                                (to-url pull-request-url "reviews")
                                {:socket-timeout 50000
                                 :connection-timeout 50000
                                 :form-params {:commit_id sha}
                                 :accept "application/vnd.github.comfort-fade-preview+json"})]
      (assoc review
        :review/state (:state resp)
        :review/id (:id resp)
        :review/node-id (:node_id resp)))
    (catch [:status 422] _
      (log/errorf "Failed to post review to %s: (%s)" pull-request-url sha))))

(defn submit-review! [{:review/keys [node-id body pull-request-url]} authed-client event]
  (log/debugf "Submitting %s review at %s" event pull-request-url)
  (api/submit-review authed-client body node-id event))

(defn delete-old-reviews!
  [authed-client pull-request-url reviewer run-tag]
  (let [classify #(case (:state %)
                    "PENDING" :delete
                    ("APPROVED" "COMMENTED") :minimize
                    :ignore)
        {:keys [delete minimize]} (->> (api/get-reviews authed-client pull-request-url reviewer)
                                       (filter #(str/starts-with? (:body %) run-tag))
                                       (group-by classify))]
    (log/error "Minimizing reviews " (with-out-str (clojure.pprint/pprint delete)))
    (api/delete-reviews authed-client (map :id delete))
    (api/minimize-reviews authed-client (map :id minimize))))

(defn delete-unused-comments!
  "Deletes any ongoing pending review and any old comments that are no longer needed.

  Comments to delete is list of node IDs"
  [authed-client prev-comments new-comments]
  (try
    (->> (keep :comment/node-id new-comments)
         (apply disj (into #{} (map :comment/node-id) prev-comments))
         (api/delete-review-comments authed-client))
    (catch Exception e
      (log/error e "Deleting review comments failed"))))

(defn add-comments-to-review!
  "Adds comments that are not on PR already (i.e. don't have id and node-id)
  to the pending review, adds node-id to the comments."
  [{:review/keys [node-id comments] :as review} authed-client]
  ;; only add review comments that are needed
  (let [node-ids (->> (remove :comment/node-id comments)
                      (map :comment/raw)
                      (api/add-review-comments authed-client node-id))]
    (assoc review :review/comments (default-from-coll comments :comment/node-id node-ids))))

(defn find-comment-ids
  "Adds REST API HTML Url to comments that only have node-id but not link."
  [{:review/keys [id comments pull-request-url] :as review} authed-client]
  (let [;; find ID of each comment, so we can link to it in Markdown
        review-comments (map rest-api->internal (api/get-review-comments authed-client pull-request-url id))
        node-id->link (zipmap (map :comment/node-id review-comments)
                              (map :comment/link review-comments))
        get-link (fn [{:comment/keys [node-id link]}] (node-id->link node-id link))]
    (assoc review
      :review/comments
      (mapv #(assoc % :comment/link (get-link %)) comments))))

(defn pull-request-url
  [{:keys [owner-login repo-slug]} pr-id]
  (to-url "/repos" owner-login repo-slug "pulls" pr-id))

(defn pull-request-diff
  "Returns pull request as a diff"
  [authed-client pull-request-url]
  (api/api-request authed-client
                   :get
                   pull-request-url
                   {:accept "application/vnd.github.v3.diff"
                    :as :text}))

(defn expand-api-comment
  "Expands a comment returned by API to possible comments we've used to post it.

  When you get comments back from GitHub API, the model is filled out with extra data which wasn't provided
  when you've posted it. E.g.: you've posted a comment with :position 1 and it comes back with :line 355, or
  you've posted a comment with no position, just :line 200, and it comes back with :position 75.

  This complicates matters of what existing comment same as the new comment immensely. Here we try to reduce
  the API comment form to possible forms that a new comment would come in."
  [api-comment]
  (mapv (partial select-keys api-comment)
        [[:body :path :side :line :startSide :startLine]
         [:body :path :side :line]
         [:body :path :position]]))

(defn match-to-ids
  "Compares previous comments (i.e. existing comments returned by GH API and the comments-to-be-posted).

  Returns new-comments but with node-id and link added if there is an existing comment on the PR that matches this.

  Result URL changes between delta runs, so we make sure to remove it before comparing.

  If an existing comment is 'outdated' it will have a position nil, which won't match new comment's
  position 1. Currently, we will let old comment get deleted and new one posted, but we need to keep an eye on
  this."
  [prev-comments new-comments]
  ;; if new comment matches one of the shape permutations of prev comment,
  ;; collect the IDs
  (let [remove-urls #(update % :body str/replace #"\(https?://[^)]+\)" "")
        raw-comment->ids
        (into {} (for [{:comment/keys [node-id link raw]} prev-comments
                       possible-key (expand-api-comment raw)]
                   [(remove-urls possible-key) #:comment {:node-id node-id :link link}]))]
    (map #(merge % (raw-comment->ids (remove-urls (:comment/raw %)))) new-comments)))

(defn clean-and-get-review-state
  "Return any existing comments in previous review, converted to internal representation.

  Removes any pending reviews that got left behind any unsuccessful runs."
  [authed-client pull-request-url run-tag]
  (let [reviewer (api/get-my-login authed-client)
        _ (delete-old-reviews! authed-client pull-request-url reviewer run-tag)]
    (->> (api/get-our-review-comments authed-client pull-request-url reviewer)
         (filter #(str/starts-with? (:body %) run-tag))
         (map rest-api->internal))))

(defn fix-single-line-comment
  "If comment is single line, then it mustn't have start+end, just line."
  [{:keys [startLine side line startSide] :as c}]
  (if (and (= startLine line) (= side startSide))
    (dissoc c :startSide :startLine)
    c))

(defn change-side-line
  "Returns side of the change and the line of the change"
  [diff]
  (cond (pos-int? (:count diff 0)) ['RIGHT (:start-line diff)]
        (pos-int? (:count-before diff 0)) ['LEFT (:start-line-before diff)]))

(defn comment-location
  "Returns comment location, path, etc.... as a github compatible map."
  [{:keys [file-name old-file-name] :as ctx} diff]
  (let [[first-change-side first-change-line] (some change-side-line diff)
        {:keys [start end region after?]} (:location ctx)]
    (cond
      (and start end region) (fix-single-line-comment
                               {:path (if after? file-name old-file-name)
                                :startLine (max (:start region) start)
                                :startSide (if after? 'RIGHT 'LEFT)
                                :side (if after? 'RIGHT 'LEFT)
                                :line (min (:end region) end)})
      ;; initially the idea was to use position=1 for lines without line data, but if annotations
      ;; are used, the annotations at start of file (line=1) expand the PR diff there and position=1 on comment
      ;; then causes the comment to be on the first line of the file. This would be great, but that "artificially"
      ;; expanded diff part is not part of any changes so GitHub marks comment to be Obsolete immediately, so it
      ;; doesn't show up on Files view (can be still seen in Conversation tab). So instead we tag the first changed
      ;; line in file.
      first-change-side {:side first-change-side
                         :line first-change-line
                         :path (if (= 'RIGHT first-change-side) file-name old-file-name)}
      :else {:position 1
             :path file-name})))

(defn new-code-health-comments
  "Code Health comments"
  [{:keys [filtered-findings] :as pr-presentable} locator]
  (let [comment-item-fn (result/file-comment-item pr-presentable locator)]
    (filtered-findings
      (fn [file-result findings cd]
        (when-let [ctx (comment-item-fn file-result findings cd)]
          (merge {:comment/raw (assoc (comment-location ctx (:diff file-result))
                                 :body (template/render-file "templates/delta/file-comment.md" ctx))}
                 (result/finding-list-item file-result findings cd)))))))

(defn review-spec [sha comments review-name body-fn]
  {:review/sha sha
   :review/comments comments
   :review/name review-name
   :review/body-fn body-fn})

(defn new-review
  [authed-client pull-request-url review-spec approve?]
  (let [{:review/keys [name comments body-fn]} review-spec
        prev-comments (clean-and-get-review-state authed-client pull-request-url (run-tag name))
        comments (mapv #(update-in % [:comment/raw :body] add-run-tag name)
                       comments)
        comments (match-to-ids prev-comments comments)]
    (delete-unused-comments! authed-client prev-comments comments)
    (-> (assoc review-spec
          :review/pull-request-url pull-request-url
          :review/body (some-> (body-fn comments)
                               (add-run-tag name))
          :review/comments comments)
        (create-review! authed-client)
        (add-comments-to-review! authed-client)
        (find-comment-ids authed-client)
        (submit-review! authed-client (if approve? "APPROVE" "COMMENT")))))
