(ns codescene.features.repository-provider.github.graphql
  (:require [clojure.string :as str]
            [codescene.url.url-utils :as url]
            [codescene.features.repository-provider.providers :as providers]
            [codescene.features.client.graphql :as gql]
            [medley.core :refer [assoc-some]]
            [taoensso.timbre :as log]
            [slingshot.slingshot :refer [try+]]))

(defn- repo-by-id [idx external-id inner-query]
  `{(~(gql/as (str "part_" idx) :node) {:id ~external-id})
    {(... on Repository) ~inner-query}})

(defn- repo-by-name [idx owner-login repo-slug inner-query]
  `{(~(gql/as (str "part_" idx) :repository) {:owner ~owner-login :name ~repo-slug})
    ~inner-query})

(def ^:private repository-projection
  '[id name isPrivate {owner [__typename login] defaultBranchRef [name]} sshUrl])

(def ^:private graphql-branches-properties
  '[{name nil
     owner [login]
     defaultBranchRef [name]
     (refs {:refPrefix "refs/heads/"
            :orderBy {:field ALPHABETICAL :direction ASC}}) {pageInfo [hasNextPage endCursor] edges {node [name]}}}])

(defn- repository-query [idx {:keys [owner-login repo-slug external-id]}]
  (if external-id
    (repo-by-id idx external-id repository-projection)
    (repo-by-name idx owner-login repo-slug repository-projection)))

(defn- repository-branch-query [per-page idx {:keys [owner-login repo-slug external-id cursor]}]
  (let [inner-query (gql/update-arguments
                      #(cond-> %2
                        per-page (assoc :first per-page)
                        cursor (assoc :after cursor))
                      graphql-branches-properties)]
    (if external-id
      (repo-by-id idx external-id inner-query)
      (repo-by-name idx owner-login repo-slug inner-query))))

(defn- response-seq [resp cnt]
  (eduction (map #(keyword (str "part_" %)))
            (map (-> resp :body :data))
            (range cnt)))

(defn repositories [query-fn repo-infos]
  (let [q (gql/generate-query [(map-indexed repository-query repo-infos)])]
    (try+
      (map (fn [{:keys [id name isPrivate sshUrl owner defaultBranchRef] :as _repo}]
             (assoc-some
               {:provider-id :github
                :private? (not (false? isPrivate))
                :accessible? (some? isPrivate)}
               :url (some-> sshUrl
                            (url/repo-url->repo-info :github false)
                            providers/repo-info->html-url
                            (str ".git"))
               :repo-slug name
               :owner-login (:login owner)
               :owner-type (some-> owner :__typename str/lower-case)
               :external-id id
               ;; this can be nil if the repo is empty
               :default-branch (:name defaultBranchRef)))
           (response-seq (query-fn q) (count repo-infos)))
      (catch [:type :http-error :status 401] _
        (log/errorf (:throwable &throw-context) "Error getting repository accessibility for %s" (pr-str repo-infos))
        repo-infos))))

(defn repository-branches
  "Fetch branch info for specified repo-infos and page cursors"
  [query-fn per-page paging-states]
  (let [query (->> paging-states
                   (map-indexed (partial repository-branch-query per-page))
                   vector gql/generate-query)]
    (map (fn [{:keys [add-page]} data]
           (if data
             (add-page (->> data :refs :edges (map (comp :name :node)))
                       (when (-> data :refs :pageInfo :hasNextPage)
                         (-> data :refs :pageInfo :endCursor))
                       {:default-branch (-> data :defaultBranchRef :name)})
             (add-page nil nil providers/no-branch-results)))
         paging-states
         (response-seq (query-fn query) (count paging-states)))))

(defn delete-review-comments
  "Constructs mutation to delete review comments."
  [node-ids]
  (gql/generate-query
    {:mutation (map-indexed (fn [idx id]
                              `{(~(gql/as (str "m" idx) :deletePullRequestReviewComment) {:input {:id ~id}})
                                {pullRequestReview [id]}})
                            node-ids)}))

(defn delete-reviews
  "Constructs mutation to delete reviews."
  [node-ids]
  (gql/generate-query
    {:mutation (map-indexed (fn [idx id]
                              `{(~(gql/as (str "m" idx) :deletePullRequestReview) {:input {:pullRequestReviewId ~id}})
                                {pullRequestReview [id]}})
                            node-ids)}))

(defn minimize-comments
  "Constructs mutation to minimize comments"
  [node-ids]
  (gql/generate-query
    {:mutation (map-indexed (fn [idx id]
                              `{(~(gql/as (str "m" idx) :minimizeComment) {:input {:classifier OUTDATED
                                                                                   :subjectId ~id}})
                                [:clientMutationId]})
                            node-ids)}))

(defn pr-query [pull-request-url inner-query]
  (let [[_ owner repo-slug pr-id] (re-find #"/([^/]+)/([^/]+)/pulls/(\d+)$" pull-request-url)]
    {:query
     `{(repository {:owner ~owner :name ~repo-slug})
       {(pullRequest {:number ~(Long/parseLong pr-id)})
        ~inner-query}}}))

(defn add-review-comments [query-fn review-node-id comments]
  (let [->parts (fn [{:keys [position] :as comment}]
                  (if position
                    [:addPullRequestReviewComment comment {:comment [:id]}]
                    [:addPullRequestReviewThread comment {:thread {'(:comments {:first 1}) {:nodes [:id]}}}]))
        mutations (map-indexed
                    (fn [idx comment]
                      (let [[mutation input output] (->parts comment)]
                        {(list (gql/as (str "m" idx) mutation)
                               {:input (assoc input :pullRequestReviewId review-node-id)})
                         output}))
                    comments)
        resp-data (-> {:mutation (vec mutations)} gql/generate-query query-fn :body :data)]
    (map (fn [idx] (let [k (keyword (str "m" idx))]
                     (or (get-in resp-data [k :thread :comments :nodes 0 :id])
                         (get-in resp-data [k :comment :id]))))
         (range (count comments)))))

(defn submit-review
  [query-fn body review-node-id event]
  (let [review (assoc-some
                 {:event event
                  :pullRequestReviewId review-node-id}
                 :body body)]
    (->> `{mutation {(submitPullRequestReview {:input ~review}) {:pullRequestReview [id]}}}
         gql/generate-query
         query-fn)))

(defn reviews
  [pull-request-url login cursor]
  (gql/generate-query
    (pr-query pull-request-url
              `{(reviews ~(cond-> {:author login :first 100}
                            cursor (assoc :after cursor)))
                {pageInfo [hasNextPage endCursor]
                 nodes [id state body]}})))

(defn teams-with-members
  [org-login end-cursor]
  (let [teams-param (if end-cursor
                      `{:first 100 :after ~end-cursor}
                      `{:first 100})]
    (gql/generate-query
     `{query {(organization {:login ~org-login})
              {(teams ~teams-param)
               {pageInfo  [endCursor hasNextPage]
                edges {node [name slug id privacy description url
                             {members
                              {edges
                               {node [login id url avatarUrl name]}}}]}}}}})))

;(teams-with-members "empear-analytics" "bob")
;(teams-with-members "bob")

(defn provider-id-for-user
  [user-login]
  (gql/generate-query `{query {(user {:login ~user-login}) [id]}}))

(defn commits-for-user
  [user-login user-id opts]
  (let [max-repos (get opts :max-repos 10)
        max-commits (min (get opts :max-commits 20) 100)]
    (gql/generate-query
      `{query {(user {:login ~user-login})
               {(repositoriesContributedTo {:first ~max-repos :contributionTypes COMMIT})
                {nodes [name {defaultBranchRef
                              {target [{(... on Commit)
                                        [{(history {:first ~max-commits :author {:id ~user-id}})
                                          {nodes [{(... on Commit) [{author [name]} messageHeadline]}]}}]}]}}]}}}})))

(comment

  (graphql-request authed-client (format
                                   "query OrgTeamsWithMembers { organization(login:\"%s\"), {
             teams(first:100) {
               edges {
                 node {
                   name slug id privacy description members {
                     edges {
                       node {
                         login id url avatarUrl name 
                       }
                     }
                   }
                 }
               }
             }
           }
      }")))
