(ns codescene.features.repository-provider.github.coverage
  (:require [codescene.features.repository-provider.github.checks :as checks]
            [codescene.features.code-coverage.pr-check :as pr-check]
            [codescene.features.repository-provider.github.review :as review]
            [taoensso.timbre :as log]))

(defn trim-body [^String text]
  ;; There's a 64kb limit by GitHub, we try to enforce here, but if they use
  ;; a lot of characters that expand to 2 or 3 bytes in UTF-8, this might not be
  ;; good enough
  (if (< 64000 (alength (.getBytes text)))
    (str (subs text 0 64000) "... OMITTED DUE TO SIZE")
    text))

(defn post-pending-run
  [auth hook-body pull-request]
  (checks/create-check-run auth
                           (->> hook-body :repository :url)
                           {:name (->> pull-request :base :ref (format "CodeScene Code Coverage (%s)"))
                            :head-commit-sha (->> pull-request :head :sha)
                            :external-id (->> pull-request :url (str "COV#"))}))

(defn post-review [authed-client pull-request-url body sha]
  (let [{:keys [output conclusion]} body
        spec (review/review-spec sha
                                 []
                                 "cs-code-coverage"
                                 (constantly (str "## " (:title output)
                                                  "\n\n" (:summary output))))]
    (review/new-review authed-client
                       pull-request-url
                       spec
                       (= conclusion "success"))))

(defn update-check-run
  [authed-client-fn metadata body sha]
  (let [{:keys [app-installation-id check-run-url]} metadata
        auth-client (authed-client-fn app-installation-id)]
    (checks/update-check-run auth-client check-run-url body)
    (post-review auth-client (:pull-request-url metadata) body sha)))

(defn result-output
  [{:keys [coverage-results] :as results}]
  (log/infof "Posting coverage results to GitHub %s" (pr-check/result-coord results))
  (let [renderable (pr-check/result->renderable coverage-results)]
    {:conclusion (if (= :fail (pr-check/check-status coverage-results)) "failure" "success")
     :output {:title (pr-check/title renderable :github)
              :summary (trim-body (str (pr-check/summary renderable :github)
                                       "\n\n"
                                       (pr-check/itemized renderable :github)))}}))

(defrecord GithubCoverageCheck [authed-client-fn]
  pr-check/PrCheckProvider
  (-provider-id [this] :github)
  (-post-result [this result metadata]
    (update-check-run authed-client-fn
                      metadata
                      (result-output result)
                      (:commit-sha result))))
