(ns codescene.features.repository-provider.github.api
  "For Github we'll be using both GraphQL and REST API.

  Beware that there limits max 100 items per page on GitHub API."
  (:require [clj-http.client :as http-client]
            [clojure.string :as str]
            [clojure.spec.alpha :as s]
            [codescene.specs :as base-specs]
            [codescene.features.client.api :as api-client :refer [to-url]]
            [codescene.features.repository-provider.github.graphql :as graphql]
            [codescene.features.repository-provider.providers :as providers]
            [codescene.features.repository-provider.specs :as specs]
            [codescene.features.util.maps :refer [map-of map-of-db ->db-keys]]
            [medley.core :refer [assoc-some]]
            [meta-merge.core :refer [meta-merge]]
            [org.clojars.roklenarcic.paginator :as page]
            [ring.util.codec :as codec]
            [taoensso.timbre :as log])
  (:import (java.util Date)))

(def ^:private pagelen 100)
(def ^:const starfox-preview "application/vnd.github.starfox-preview+json")

(defn- process-exception
  [exception {:keys [url] :as params}]
  (let [{:keys [body status]} (ex-data exception)
        parsed-body (api-client/json-safe-parse body)
        message (:message parsed-body (ex-message exception))
        data (map-of url message)]
    (log/debugf "Github call returned %s %s" status body)
    (log/info exception "Github call exception" (api-client/printable-req params))
    (case status
      401 (api-client/ex-unauthorized :github exception (assoc data :token-expired? true))
      403 (if (and message (str/includes? message "SAML enforcement"))
            (api-client/ex-unauthorized :github exception (assoc data :token-expired? true))
            (api-client/ex-forbidden :github exception data))
      404 (api-client/ex-forbidden :github exception data)
      (api-client/ex-http-error :github exception (or status 0) data))))

(s/fdef api-request*
        :args (s/cat :auth-token ::specs/auth-token
                     :method keyword?
                     :url ::base-specs/non-empty-string
                     :params map?))
(defn api-request*
  "Core function for API requests. Requires ApiClient token.

  Uses augment-request to add some sort of credentials to the request.
  If request returns 401 then on-expired-token will be called on auth-token.

  If that returns a ApiClient, the request will be retried."
  [authed-client method url request-params]
  (let [params (meta-merge
                 {:url url
                  :accept "application/vnd.github.v3+json"
                  :follow-redirects true
                  :as :json
                  :content-type :json
                  :request-method method}
                 request-params)
        final-params (-> authed-client
                         (api-client/augment-request params)
                         (api-client/fix-api-url "https://api.github.com"))]
    (when-not (:connection-manager final-params)
      (log/warnf "No connection manager, url=%s, avoid for production use." (:url final-params)))
    (try
      (http-client/request final-params)
      (catch Exception e
        (-> e
            (process-exception final-params)
            (api-client/retry-token authed-client)
            (api-request* method url request-params))))))

(defn api-request [authed-client method url request-params]
  (:body (api-request* authed-client method url request-params)))

(defn graphql-assert-success!
  "Parse GraphQL response and if there were errors, throw a Bad Request exception."
  [resp]
  (when-let [errors (-> resp :body :errors)]
    (throw
      (api-client/service-exception
        :github
        (str/join "\n" (map :message errors))
        {:type :http-error
         :status 400
         :message (str/join "\n" (map :message errors))
         :errors errors})))
  resp)

(defn- as-long [maybe-str]
  (cond-> maybe-str (string? maybe-str) parse-long))

(defn log-rate-limit [{:keys [headers] :as resp}]
  (let [remaining (as-long (get headers "X-RateLimit-Remaining"))
        total (as-long (get headers "X-RateLimit-Limit"))
        reset-date (some-> (get headers "X-RateLimit-Reset") as-long ^long (* 1000) (Date.))]
    (cond
      (nil? remaining) (log/warnf "GitHub Rate Limit - missing 'X-RateLimit-Remaining' header")
      (< remaining 100) (log/warnf "GitHub Rate Limit - APPROACHING THE LIMIT: (remaining/total) %s/%s resets %s" remaining total reset-date)
      :else (log/infof "GitHub Rate Limit: (remaining/total) %s/%s resets %s" remaining total reset-date))
    resp))

(s/fdef graphql-request
        :args (s/cat :auth-token ::specs/auth-token
                     :throw-on-error? (s/? boolean?)
                     :query string?))
(defn graphql-request
  "Core function for GraphQl requests.  Requires ApiClient token.

  Make sure to check out GitHub GraphQL API explorer: https://developer.github.com/v4/explorer/"
  ([auth-token query] (graphql-request auth-token true query))
  ([auth-token throw-on-error? query]
   (let [params (meta-merge
                  {:url "/graphql"
                   :content-type :json
                   :as :json
                   :accept starfox-preview                  ;; necessary for github projects queries
                   :socket-timeout 50000
                   :connection-timeout 50000
                   :request-method :post
                   :form-params {:query query}})
         final-params (-> (api-client/augment-request auth-token params)
                          (api-client/fix-api-url "https://api.github.com")
                          (update :url str/replace "/v3/graphql" "/graphql"))]
     (when-not (:connection-manager final-params)
       (log/warnf "No connection manager, url=%s, avoid for production use." (:url final-params)))
     (log/debugf "Running GraphQL query %s" query)
     (cond-> (try
               (log-rate-limit (http-client/request final-params))
               (catch Exception e
                 (-> (process-exception e params)
                     (api-client/retry-token auth-token)
                     (graphql-request throw-on-error? query))))
       throw-on-error? graphql-assert-success!))))

(defn github-user-by-id
  "Get info about a user based on the id (not username).
  Uses an undocumented feature of the /user endpoint (not /users)
  actually accepting an id, as in /user/:id, eg /user/123456.
  See https://stackoverflow.com/a/30579888/1184752.
  NOTE: When the underlying token is nil, the API calls are **heavily rate limited** (60 requests / hour)!!!"
  [authed-client user-id & [options]]
  (api-request authed-client :get (to-url "user" user-id) options))

(defn api-page-request
  "A helper function for a request that requests a particular page. It is based on a paging state
  and returns an items+cursor map."
  [authed-client {:keys [cursor add-page pages]} url request-params]
  (log/debugf "Req page %s of %s with %s" pages url (codec/form-encode request-params))
  (let [params (assoc-in request-params [:query-params :per_page] 100)
        resp (if cursor
               (api-request* authed-client :get cursor (dissoc params :query-params))
               (api-request* authed-client :get url params))]
    (add-page (get-in resp (:items params [:body]))
              (get-in resp [:links :next :href]))))

(defn get-all-pages
  "Returns all pages, special param :items is a vector of where the list of items is,
  defaults to :body"
  [authed-client url params]
  (-> #(api-page-request authed-client % url params)
      page/paginate-one!
      page/unwrap))

(defn get-user
  [authed-client id]
  (github-user-by-id authed-client id {}))

(defn get-user-by-username
  [authed-client username]
  (api-request authed-client :get (str "/users/" username) {}))

(defn get-my-info
  [authed-client]
  (api-request authed-client :get "/user" {}))

(defn get-my-login
  "This uses GraphQL and works even with App tokens"
  [authed-client]
  (let [ret (graphql-request authed-client "query { viewer { login}}")]
    (-> ret :body :data :viewer :login)))

(defn get-emails
  [authed-client]
  (api-request authed-client :get "/user/emails" {}))

(defn- get-repos*
  [authed-client type pagination]
  (api-request* authed-client
                :get
                "/user/repos"
                {:query-params (merge {:type type} (->db-keys pagination))}))

(defn- get-org-repos*
  [authed-client org-login type pagination]
  (api-request* authed-client
                :get
                (to-url "/orgs" org-login "repos")
                {:query-params (merge {:type type} (->db-keys pagination))}))

(defn get-org-memberships
  "Add all pages"
  [authed-client]
  (get-all-pages authed-client
                 (to-url "/user/memberships/orgs")
                 {:query-params {:state "active"}}))

(defn get-org
  [authed-client org-login]
  (api-request authed-client
               :get
               (to-url "/orgs" org-login)
               {}))

(defn get-org-members
  "Fetches all organization's members optionally filtering them by role.
  If no role is specified then all members are returned."
  [authed-client org-login member-role]
  (get-all-pages authed-client
                 (to-url "/orgs" org-login "members")
                 {:query-params (assoc-some {} :role member-role)}))

(defn get-org-member
  "Get a membership of given user in given organization.
  Returns response body (membership data).
  Returns  nil if the user isn't the member of the org.
  Throws if the authentication token doesn't have access to given organization.
  See https://docs.github.com/en/rest/orgs/members?apiVersion=2022-11-28#get-organization-membership-for-a-user

  IMPORTANT: the GitHub API returns members in BOTH 'active' and 'pending' state as described in the API docs:
  https://docs.github.com/en/rest/orgs/members?apiVersion=2022-11-28#list-organization-memberships-for-the-authenticated-user
  By default, we only consider 'active' members and return `nil` if the member is in the 'pending' status.
  If you want to include pending invitations then pass `true` as the optional 4th arg.

  NOTE: there's also separate endpoint (https://docs.github.com/en/rest/orgs/members?apiVersion=2022-11-28#check-organization-membership-for-a-user)
  for pure membership check but that returns 204, giving us no data about the member (we need at least the id)."
  ([authed-client org-login member-username]
   (get-org-member authed-client org-login member-username false))
  ([authed-client org-login member-username include-pending?]
   (let [{:keys [body status] :as response}
         (api-request* authed-client :get (to-url "/orgs" org-login "memberships" member-username)
                       ;; don't fail on 404
                       {:throw-exceptions false})]
     (case status
       200 (if include-pending? body (when (= "active" (:state body))
                                       body))
       404 nil
       403 (throw (ex-info "Invalid token - the token owner isn't a member of the organization."
                           {:org-login org-login}))
       (throw (ex-info "Unexpected error when getting user org membership" {:org-login org-login
                                                                            :member-username member-username
                                                                            :response response}))))))


(defn- org-teams-service-exception
  [{:keys [status body] :as resp}]
  (let [errors (:errors body)
        error-str (str/join "\n" (map :message errors))]
    (api-client/service-exception :github
                                  error-str
                                  {:type :http-error
                                   :status status
                                   :message error-str
                                   :errors errors})))


(defn get-org-teams-with-members
  ([authed-client org-login]
   (get-org-teams-with-members authed-client org-login [] nil))
  ([authed-client org-login found end-cursor]
;   (def ac authed-client)
   (let [{:keys [status body] :as resp}
         (graphql-request authed-client (graphql/teams-with-members org-login end-cursor))
         {teams-edges :edges page-info :pageInfo} (-> body :data :organization :teams)
         {has-next-page? :hasNextPage new-end-cursor :endCursor} page-info]
     (cond (not= 200 status)
           (throw (org-teams-service-exception resp))

           has-next-page?
           (recur authed-client org-login (concat found teams-edges) new-end-cursor)

           :else
           (concat found teams-edges)))))

(defn- get-user-id
  [authed-client user-login]
  (let [{:keys [body status]}  (graphql-request authed-client (graphql/provider-id-for-user user-login))]
    (if (= 200 status)
      (-> body :data :user :id)
      (do
        (log/warnf "Graphql query failed with status %s when fetching user id for user %s" status user-login)
        nil))))

(defn sort-by-frequency
  [items]
  (->> items
       frequencies
       (sort-by second)
       reverse
       (mapv first)))

(defn get-commit-author-names-for-provider-user
  "Returns a list of author names used by the `user-login` Git provider
  user, searching across the 20 latest repos they have contributed to,
  inspecting 100 commits from each repo. 

  The list that is returned is a list of distinct author names, sorted
  by frequency, with the most common names associated with
  `user-login` appearing first."
  [authed-client user-login]
  (if-let [user-id (get-user-id authed-client user-login)]
    (let [{:keys [body status]} (graphql-request authed-client (graphql/commits-for-user user-login user-id {:max-repos 20 :max-commits 100}))]
      (if (= 200 status)
        (let [repos (-> body :data :user :repositoriesContributedTo :nodes)]
          {:author-matches (->> repos
                                (mapcat (fn [r] (-> r :defaultBranchRef :target :history :nodes)))
                                (map (fn [c] (-> c :author :name)))
                                sort-by-frequency)
           :status 200 :message "ok"})
        (do
          (log/warnf "Graphql query failed with status %s when fetching commits for GitHub user %s" status user-login)
          {:author-matches [] :status 500 :message "Github query for searching commits failed."})))
    ;; Unlikely, because any publicly available GH account will work here.
    {:author-matches [] :status 404 :message (format "User %s not found." user-login)}))

;(get-commit-author-names-for-provider-user ac "josf")

(comment
  (graphql-request ac
                   "query 
                  { user(login: \"josf\") 
                     { name repositoriesContributedTo(first: 10, contributionTypes: COMMIT) {
                       nodes {
                           name 
                           defaultBranchRef {
                              target {
                                ... on Commit {
                                history(first: 20, author: {id: \"MDQ6VXNlcjEwMDMxMw==\"}) {
                                 nodes {
                                  ... on Commit {
                                     author { name }
                                     messageHeadline
                                   }
                                 }
                               }
                              }
                            }
                          }
                         }
                       }

                  }}")
                                        ;
  )

(defn get-org-team-members
  [authed-client org-login team-slug]
  (let [{:keys [body status] :as response}
        (api-request* authed-client :get (to-url "/orgs" org-login "teams" team-slug "members")
                      {:throw-exceptions false})]
    ;; TODO @josf: handle error states
    body))

(defn valid-token?
  "Returns true if auth-token is valid"
  [->authed-client client-id client-secret access-token]
  (let [resp
        (api-request* (->authed-client [client-id client-secret])
                      :post
                      (to-url "/applications" client-id "token")
                      {:unexceptional-status #(<= 200 % 499)
                       :form-params (map-of-db access-token)})]
    (when (= (:status resp) 200) (:body resp))))

(defn revoke-app-grant
  [->authed-client client-id client-secret access-token]
  (let [resp
        (api-request* (->authed-client [client-id client-secret])
                      :delete
                      (to-url "/applications" client-id "grant")
                      {:throw-exceptions false
                       :form-params (map-of-db access-token)})]
    (when (= (:status resp) 200) (:body resp))))

(defn get-commits
  "Fetches all the commits from the absolute(!) commits url"
  [authed-client commits-url]
  (get-all-pages authed-client commits-url {}))

(defn get-commit-ids
  [authed-client commits-url]
  (map :sha (get-commits authed-client commits-url)))

(defn get-education-status
  "Returns GitHub education status for a github user represented by the given access token
   using special api hosted at https://education.github.com/api/user.

   Expected to return {:student true/false :faculty true/false}"
  [authed-client]
  (api-request authed-client
               :get
               "https://education.github.com/api/user"
               {:headers {"faculty-check-preview" "true"}}))

(defn- parse-page-number
  "Parses page number from the page link represented as string like this:
    https://api.github.com/user/repos?type=%3Aall&per_page=100&page=7.
  Throws NFE if page value isn't a number.
  Returns nil if the 'page' param not present in the link."
  [page-link]
  (when-let [page-param-value (-> (codec/form-decode page-link)
                                  (get "page"))]
    (Integer/valueOf ^String page-param-value)))

(defn get-repos-page [authed-client org-login page-num]
  (let [params (if page-num {:page page-num :per-page pagelen} {:per-page pagelen})
        resp (if org-login
               (get-org-repos* authed-client org-login "all" params)
               (get-repos* authed-client "all" params))]
    (with-meta (:body resp) {:pages (or (some-> resp :links :last :href parse-page-number) 1)})))

(defn get-repos
  "Fetches all available repositories page-by-page.

  First page is fetched first to determine total number of pages and then subsequent
  pages are fetched in concurrently using up to `max-concurrent-requests` requests.
  Returns a flat sequence containing all the repositories."
  [authed-client org-login max-concurrent-requests]
  (let [req-fn (page/async-fn (partial get-repos-page authed-client org-login) max-concurrent-requests)]
    (let [first-page (page/unwrap-async (req-fn nil))
          results (mapv req-fn (range 2 (inc (-> first-page meta :pages))))]
      (reduce #(into %1 (page/unwrap-async %2)) (vec first-page) results))))

(defn get-specified-repos
  "Selects basic data for several repositories. It will use external ID if possible,
  otherwise it will fall back on owner login + repository name."
  [authed-client repo-infos]
  (mapcat (partial graphql/repositories #(graphql-request authed-client false %))
          (partition-all pagelen repo-infos)))

(defn get-repos-branches
  [authed-client repo-infos max-concurrent-requests max-branches]
  (let [query-fn #(graphql-request authed-client false %)
        get-branches #(->>
                        (map (fn [state] (api-client/limit-items state max-branches)) %)
                        (graphql/repository-branches query-fn pagelen))]
    (->> repo-infos
         ;; will return paging-states instead of items, but that allows String items
         (page/paginate! (page/async-fn get-branches max-concurrent-requests) {:batcher 50})
         (mapv providers/->branch-results))))

(defn create-comment
  [authed-client comments-url text]
  (:id (api-request authed-client
                    :post
                    comments-url
                    ;; TODO ROK This is just for onprem users with GitHub enterprise server that is old. This preview has graduated
                    {:accept "application/vnd.github.antiope-preview+json"
                     :form-params {:body text}})))

(defn delete-review-comments
  [authed-client comment-node-ids]
  (doseq [batch (partition-all 15 comment-node-ids)
          :when (seq batch)]
    (try (graphql-request authed-client false (graphql/delete-review-comments batch))
         (catch Exception e
           (log/error e "Failed to delete reviews comments")))))

(defn delete-reviews
  [authed-client review-node-ids]
  (log/debug "Deleting pending reviews " (vec review-node-ids))
  (doseq [batch (partition-all 15 review-node-ids)]
    (graphql-request authed-client (graphql/delete-reviews batch))))

(defn minimize-reviews
  [authed-client review-node-ids]
  (log/debug "Minimize reviews " (vec review-node-ids))
  (doseq [batch (partition-all 15 review-node-ids)]
    (graphql-request authed-client (graphql/minimize-comments batch))))


(defn get-reviews
  [authed-client pull-request-url login]
  (loop [cursor nil
         reviews []]
    (let [q (graphql/reviews pull-request-url login cursor)
          resp (get-in (graphql-request authed-client q) [:body :data :repository :pullRequest :reviews])]
      (if (-> resp :pageInfo :hasNextPage)
        (recur (-> resp :pageInfo :endCursor) (into reviews (:nodes resp)))
        (into reviews (:nodes resp))))))

(def review-comment-keys [:body :path :side :line :startSide :startLine :position])

(defn add-review-comments
  "Adds comments to the review, return node ids"
  [authed-client review-node-id comments]
  (->> (partition-all 12 comments)
       (mapcat
         (fn [comment-batch]
           (graphql/add-review-comments #(graphql-request authed-client %) review-node-id comment-batch)))
       doall))

(defn submit-review
  [authed-client body review-node-id event]
  (graphql/submit-review #(graphql-request authed-client %) body review-node-id (symbol event)))

(defn host->api-url [host]
  (when (and host (and (not= host "github.com")
                       (not= host "api.github.com")))
    (format "https://%s/api/v3" host)))

(defn get-our-review-comments
  [authed-client pull-request-url my-login]
  (let [all-comments (get-all-pages authed-client (to-url pull-request-url "comments") {})]
    (filter #(-> % :user :login (= my-login)) all-comments)))

(defn get-review-comments
  "Returns all comments on the review. This is expected to be used on a pending review, so
  all comments are ours."
  [authed-client pull-request-url review-id]
  (get-all-pages authed-client (to-url pull-request-url "reviews" review-id "comments") {}))
