(ns codescene.features.repository-provider.azure.delta
  "Azure DevOps delta implementation uses Webhooks (or Service Hooks as they are called).
  This requires that we install/uninstall them, similarly to GitLab, but Azure at least has a special
  HTTP code you can return to disable the webhook.

  If error response is returned for webhook, your webhook will be put on probation and you'll get a payload only every
  couple of minutes until you have a few successful returns. That causes the webhook to miss all kinds of events
  which is not acceptable. This is why it's important to catch errors and return 200 responses.

  The webhooks are separate for create and update of PR, so we have two webhooks. You can find hooks
  in https://dev.azure.com/codescene-cloud/azure-integration/_settings/serviceHooks project settings page.
  You can inspect them and hook history. Hooks can be configured with custom headers. We add a header
  with a salted token like GitLab, which is checked on the webhook.

  The results are reported via 2 mechanisms:
  - pull request comments
  - build checks

  The pull request comments contain a tag that marks them as CodeScene comment
   and it helps identify which environment they're from. Same token-secret = same environment.
  For each analysis the old comments are deleted and a new comment is created. If delta analysis passes,
  then the comment is posted with Resolved status, otherwise it's Active. This enables that client can
  put a quality gate on it via a rule that all comments on PR must be resolved before merge.

  We also post a build check which can be used as quality gate."
  (:require [clojure.string :as str]
            [codescene.features.delta.protocols :as protocols]
            [codescene.features.project.pr-integration :as pi]
            [codescene.features.repository-provider.azure.api :as api]
            [codescene.features.delta.file-regions :as file-regions]
            [codescene.features.delta.result.pull-request :as delta-result]
            [codescene.features.repository-provider.providers :as providers]
            [codescene.features.util.coll :as u.coll]
            [codescene.features.security.tokens :as tokens]
            [codescene.features.util.template :as template]
            [com.climate.claypoole :as cp]
            [org.clojars.roklenarcic.paginator :as page]
            [taoensso.timbre :as log]))

(defn ->run-tag [pr-app-id]
  ; identifies specific check's comments by our server's secret
  (format "\n<!-- %s -->" pr-app-id))

(defn- thread-fn
  "Return thread request data, expects that diffs have been converted locators."
  [active-status? {:keys [location change-level file-name] :as comment-context}]
  (let [{:keys [start end after?]} location
        start-pos {:offset 1 :line (some-> start (max 1))}
        end-pos {:offset 1 :line (some-> end (max 1))}
        sides (if after?
                {:rightFileStart start-pos :rightFileEnd end-pos}
                {:leftFileStart start-pos :leftFileEnd end-pos})]
    {:comments [{:content (template/render-file "templates/delta/file-comment.md" comment-context)}]
     :status (case change-level
               :warning (if active-status? "active" "closed")
               :improvement "fixed"
               :notice "closed")
     :threadContext (merge {:filePath (str "/" file-name)}
                           (when (and start end)
                             sides))}))

(defn threads [active-status? pr-presentable locator]
  (->> (delta-result/file-comment-items pr-presentable locator)
       (map #(thread-fn active-status? %))))

(defn codescene-thread?
  [run-tag thread]
  (str/includes? (-> thread :comments first :content) run-tag))

(defn thread-key
  "Select key properties of thread to determine if it's the same logical comment."
  [{:keys [comments threadContext]}]
  {:threadContext threadContext
   :comments [(select-keys (first comments) [:content])]})

(defn get-our-threads [authed-client url run-tag]
  (->> (api/get-pull-request-threads authed-client url)
       (remove (comp :isDeleted first :comments))
       (filter (partial codescene-thread? run-tag))))

(defn post-result
  "Updates build status, deletes old pull request
  comments and creates a new comment.

  status is one of :success :failure :skip-success :skip-fail :error :pending"
  [authed-client-fn provider-ref {:keys [comment extra-threads status status-text result-url]}]
  (log/debugf "provider=azure; action=delta-analysis; ref=%s"
              provider-ref)
  (let [{:keys [pr-app-id url target-branch status-name result-options]} provider-ref
        {:keys [high-risk-active-comment?]} result-options
        ; identifies specific check's comments by hook secret
        run-tag (->run-tag pr-app-id)
        add-run-tag #(update-in % [:comments 0 :content] str run-tag)
        extra-threads (mapv add-run-tag extra-threads)
        authed-client (authed-client-fn provider-ref)
        pr-comment-status (case status
                            (:success :skip-success) "fixed"
                            :high-risk-success (if high-risk-active-comment? "active" "fixed")
                            (:failure :error :skip-fail :pending) "active")
        pr-status {:status-code (case status
                                  (:success :high-risk-success) "succeeded"
                                  :pending "pending"
                                  (:failure :skip-fail) "failed"
                                  :error "error"
                                  :skip-success "notApplicable")
                   :target-url result-url
                   :status-name (str "(-> " target-branch ")")
                   :description (str/replace status-text \newline \space)}
        {:keys [removed added]} (u.coll/do-diff thread-key
                                                (get-our-threads authed-client url run-tag)
                                                extra-threads)]
    (doseq [thread removed
            ;; don't delete prev comments when posting a pending status
            :when (not= status :pending)]
      (api/delete-pull-request-comment authed-client thread))
    ;; UI displays comments in reverse order, we need to post the overall comment last, so it appears at the top
    (doseq [thread (reverse added)
            :when comment]
      (api/add-pull-request-thread authed-client url thread))
    (when comment
      (api/add-pull-request-thread authed-client url
                                   (add-run-tag {:comments [{:content comment}]
                                                 :status pr-comment-status})))
    (api/add-pull-request-status authed-client url pr-status (or status-name "CodeScene Delta Analysis"))))

(defn post-delta-results [delta-authed-client-fn {:keys [result-options] :as provider-ref} pr-presentable]
  (let [negative-review? (delta-result/negative-review? pr-presentable)
        high-risk? (-> pr-presentable :result :high-risk?)
        locator (delta-result/cached-locator #(comp first (file-regions/change-locator % false)))
        {:keys [annotations? active-status?]} result-options
        result-options (assoc result-options :include-review-findings? annotations?)
        comment (when (delta-result/comment? pr-presentable result-options)
                  (delta-result/presentable->comment pr-presentable :azure (not annotations?)))]
    (post-result
      delta-authed-client-fn
      provider-ref
      {:comment comment
       :extra-threads (when annotations? (threads active-status? pr-presentable locator))
       :status (cond
                 negative-review? :failure
                 high-risk? :high-risk-success
                 :else :success)
       :result-url (:result-url pr-presentable)
       :status-text (str "Code Health Quality Gates: " (if negative-review? "FAILED" "OK"))})))

(defn post-delta-skipped [delta-authed-client-fn provider-ref reason result-url]
  (let [comment-title (delta-result/render-error-title reason)
        status (if (providers/negative-skip? reason) :skip-fail :skip-success)]
    (post-result delta-authed-client-fn
                 (assoc provider-ref :result-url result-url)
                 {:comment (when (or (-> provider-ref :result-options :always-comment?) (= :skip-fail status))
                             (format "### CodeScene PR Check Skipped\n%s\n---\n%s"
                                     comment-title
                                     (delta-result/render-error reason)))
                  :status status
                  :status-text comment-title})))

(defn post-delta-error [delta-authed-client-fn provider-ref error result-url]
  (post-result delta-authed-client-fn
               provider-ref
               {:comment (str "### CodeScene PR Check\n"
                              (delta-result/render-error-title error)
                              "\n---\n"
                              (delta-result/render-error error))
                :status :error
                :status-text (delta-result/render-error-title error)
                :result-url result-url}))

(defn post-delta-pending [delta-authed-client-fn provider-ref result-url]
  (post-result delta-authed-client-fn
               provider-ref
               {:comment "### CodeScene PR Check Pending"
                :status :pending
                :status-text "PR Check Pending"
                :result-url result-url}))

(defrecord AzureDelta [delta-authed-client-fn]
  protocols/DeltaResultBoundary
  (delta-pending [this provider-ref result-url]
    (post-delta-pending delta-authed-client-fn provider-ref result-url))
  (delta-results [this provider-ref pr-presentable]
    (post-delta-results delta-authed-client-fn provider-ref pr-presentable))
  (delta-skipped [this provider-ref reason result-url]
    (post-delta-skipped delta-authed-client-fn provider-ref reason result-url))
  (delta-error [this provider-ref error result-url]
    (post-delta-error delta-authed-client-fn provider-ref error result-url)))

(defn webhook-status
  [pr-integration hooks]
  (cond
    (and (= 2 (count hooks))
         (every? :enabled? hooks)
         (every? #(pi/hook-token-verify pr-integration %) hooks))
    :installed
    (zero? (count hooks))
    :not-installed
    :else
    :invalid))

(defn repository-webhook-status
  [pr-integration repositories concurrency]
  (let [our-hook? (pi/our-hook-fn? pr-integration)
        authed-client (pi/-authed-client pr-integration (first repositories))
        repositories (if (not-any? :external-id repositories)
                       (api/with-external-ids authed-client repositories)
                       repositories)
        external-id->hooks (->> (api/get-accounts-webhooks authed-client repositories our-hook? concurrency)
                                (mapcat page/unwrap)
                                (group-by :external-id))]
    ;; keep the order by doing this
    (mapv (fn [{:keys [external-id]}]
            (let [hooks (external-id->hooks external-id)]
              {:hooks hooks
               :status (webhook-status pr-integration hooks)}))
          repositories)))

;;
(defrecord AzureHookAccess [concurr-limit]
  protocols/HookAccess
  (-repos-hooks [this PrIntegrationConfig repositories]
    (repository-webhook-status PrIntegrationConfig repositories concurr-limit))
  (-add-hooks [this PrIntegrationConfig repositories]
    (cp/pdoseq concurr-limit
               [repo repositories]
               (log/infof "Installing Azure webhook to %s" (:url repo))
               (api/add-webhook
                 (pi/-authed-client PrIntegrationConfig repo)
                 repo
                 (pi/-callback-url PrIntegrationConfig)
                 (cond->> (pi/-hook-secret PrIntegrationConfig)
                   (:external-id repo) (tokens/b64-salted-token (:external-id repo))))))
  (-remove-hooks [this PrIntegrationConfig repos]
    (cp/pdoseq concurr-limit
               [repo repos
                :let [authed-client (pi/-authed-client PrIntegrationConfig repo)]
                hook (:hooks repo)]
               (log/infof "Removing Azure webhook from %s" (:url hook))
               (api/remove-webhook authed-client (:url hook)))))