(ns codescene.features.reports.portfolio.status-overview
  (:require
   [codescene.features.reports.pdf-helpers :as ph]
   [codescene.features.reports.graphics.functions :as graphics]
   [codescene.features.reports.common.pdf-elements :as pdf-elements]
   [codescene.features.reports.common.styling :as styling]
   [codescene.features.presentation.scoring :as scoring]))

(defn- graphics-table [{:keys [risky attention healthy]}]
  (ph/table
   [:so-graphics-table] [90]
   (ph/row
    (ph/cell
     [:so-graphics-cell]
     {:background-layer-fn
      (ph/draw-using-g2d-fn
       (graphics/code-health-overview-doughnut-fn
        {:red-value risky :yellow-value attention :green-value healthy}))}))))


(defn- info-with-score-color [score-cell-style title [current-period previous-period] img-dir]
  (let [change (- (or current-period 0) (or previous-period 0))
        change-icon (cond
                      (neg? change) (styling/load-resource img-dir "period-stats/Arrow-down-circle.png")
                      (pos? change) (styling/load-resource img-dir "period-stats/Arrow-up-circle.png")
                      :else (styling/load-resource img-dir "period-stats/No-change-circle.png"))
        change-text (cond
                      (neg? change) (styling/number->printable-string-100k change)
                      (pos? change) (format "+%s" (styling/number->printable-string-100k change))
                      :else "No change")
        current-period (if (> current-period 999) ">1k" current-period)]
    (ph/cell
     [:c-leftborder-cell score-cell-style]
     (ph/paragraph [:c-subheader] title)
     (ph/table
      [:so-info-table] [3 2 15]
      (ph/row
       (ph/cell [:so-info-cell] (ph/paragraph [:so-number-paragraph] current-period))
       (ph/cell [:so-info-cell] (ph/image [:so-changeicon-image] change-icon))
       (ph/cell [:so-info-cell] (ph/paragraph [:so-changetext-paragraph] change-text)))))))


(defn- factor-title-and-info-table [title img-dir num-projects num-results]
  (let [[info-text info-icon] (if (= num-projects num-results)
                                [(format "Status available for all %d projects" num-projects)
                                 (styling/load-resource img-dir "portfolio/Check.png")]
                                [(format "Status available for %d/%d projects. This factor is not configured on all projects."
                                         num-results num-projects)
                                 (styling/load-resource img-dir "portfolio/Alert-octagon.png")])]
    (ph/table [] [12 1 23]
              (ph/row (ph/cell [:so-info-cell] (ph/paragraph [:so-factor-title-paragraph] title))
                      (ph/cell [:so-info-cell] (ph/image [:so-info-icon] info-icon))
                      (ph/cell [:so-info-cell] (ph/paragraph [:so-info-text] info-text))))))


(defn- factor-status-table
  [title img-dir {:keys [now before num-projects num-results]}]
  (let [healthy-values (map :healthy [now before])
        attention-values (map :attention [now before])
        risky-values (map :risky [now before])]
    (ph/table
     [] [3 20]
     (ph/row (ph/cell [:so-lefthand-cell] (graphics-table now))
             (ph/cell [:so-righthand-cell]
                      (factor-title-and-info-table title img-dir num-projects num-results)
                      [:spacer 1]
                      (ph/table [] nil
                                (ph/row
                                 (info-with-score-color :c-border-healthy "Healthy status" healthy-values img-dir)
                                 (info-with-score-color :c-border-attention "Attention status" attention-values img-dir)
                                 (info-with-score-color :c-border-risky "Risky status" risky-values img-dir))))))))

(defn empty-state-table [title text link-table img-dir]
  (let [mock-diagram (styling/load-resource img-dir "portfolio/Pie chart.png")]
    (ph/table
     [:so-na-table] [3 20]
     (ph/row (ph/cell [] (ph/image [:so-na-image] mock-diagram))
             (ph/cell [:so-na-righthand-cell]
                      (ph/paragraph [:so-factor-title-paragraph] title)
                      (ph/paragraph [:so-na-info-text] text)
                      link-table))))
  )

(defn- team-code-alignment-part [team-code-alignment-results? team-code-alignment
                                 {:keys [img-dir global-teams-conf-url]}]
  (if team-code-alignment-results?
    (factor-status-table "Team-code Alignment" img-dir team-code-alignment)
    (empty-state-table "Team-code Alignment"
                       "No data to show due to missing or incomplete configurations. The Configuration is set up per project."
                       (ph/table [] [3 18] (pdf-elements/link-row global-teams-conf-url "Define teams" img-dir))
                       img-dir)))

(defn- delivery-part [delivery-performance-results? delivery-performance?
                      delivery-performance {:keys [img-dir delivery-config-docs-url upgrade-plan-url]}]
  (cond (not delivery-performance?)
        (empty-state-table "Delivery"
                           "Delivery is an output metric that captures the efficiency of the overall development process. To see Delivery metrics, please upgrade to a Pro plan.  "
                           (ph/table [] [4 24] (pdf-elements/link-row upgrade-plan-url "Upgrade Plan" img-dir))
                           img-dir)
        (not delivery-performance-results?)
        (empty-state-table "Delivery"
                           "No data to show due to missing or incomplete configurations. The Configuration is set up per project."
                           (ph/table [] [4 17] (pdf-elements/link-row delivery-config-docs-url "Configure PM tool" img-dir))
                           img-dir)
        :else
        (factor-status-table "Delivery" img-dir delivery-performance)))


(defn- ->score-level-count [trend item-fn factor-key factor-fn]
  (let [grouped-levels (->> trend
                            (map item-fn)
                            (map factor-fn)
                            (map #(scoring/score-info-for % factor-key))
                            (group-by :score))]
    {:healthy (count (get grouped-levels 2))
     :attention (count (get grouped-levels 1))
     :risky (count (get grouped-levels 0))}))

(defn- factor-scores [factor-key factor-fn projects]
  (let [trend (->> projects
                   (map :renderable-trends)
                   (map factor-key))
        score-levels-now (->score-level-count trend first factor-key factor-fn)
        score-levels-before (->score-level-count trend last factor-key factor-fn)
        num-results (reduce + (vals score-levels-now))]
    {:before score-levels-before
     :now score-levels-now
     :num-results num-results
     :num-projects (count projects)}))


(defn ->factor-statuses [projects]
  (let [ch-scores (factor-scores :code-health :weighted-kpi projects)
        kd-scores (factor-scores :knowledge-distribution :value projects)
        tc-scores (factor-scores :team-code-alignment :value projects)
        dp-scores (factor-scores :delivery-performance :value projects)]
    {:code-health ch-scores
     :knowledge-distribution kd-scores
     :team-code-alignment tc-scores
     :delivery-performance dp-scores}))



(defn pdf-element [projects {:keys [delivery-performance?]} {:keys [img-dir projects-uri] :as options}]
  (let [{:keys [code-health knowledge-distribution team-code-alignment delivery-performance]} (->factor-statuses projects)
        header-text (format "4 FACTOR STATUS OVERVIEW OF YOUR %d PROJECTS" (count projects))
        team-code-alignment-results? (> (:num-results team-code-alignment) 0)
        delivery-performance-results? (> (:num-results delivery-performance) 0)]
    (ph/table
     [:c-card-content-top-table] nil
     (ph/row
      (ph/table
       [] nil
       (ph/row
        (ph/cell [] (ph/paragraph [:c-miniheader] header-text) [:spacer 2])
        (ph/cell [:so-link-cell] (ph/table [] [10 1] (pdf-elements/link-row projects-uri "See Detailed View" img-dir))))))
     (ph/row (ph/cell [] (factor-status-table "Code Health" img-dir code-health)))
     (ph/row (ph/cell [:so-line-cell] (ph/line [:c-line])))
     (ph/row (ph/cell [] (factor-status-table "Knowledge Distribution" img-dir knowledge-distribution)))
     (if team-code-alignment-results?
       (ph/row (ph/cell [:so-line-cell] (ph/line [:c-line])))
       (ph/row (ph/cell [:so-na-line-cell] (ph/line [:c-line]))))
     (ph/row (ph/cell [] (team-code-alignment-part team-code-alignment-results? team-code-alignment options)))
     (if (and delivery-performance? delivery-performance-results?)
       (ph/row (ph/cell [:so-line-cell] (ph/line [:c-line])))
       (ph/row (ph/cell [:so-na-line-cell] (ph/line [:c-line]))))
     (ph/row (ph/cell [] (delivery-part delivery-performance-results? delivery-performance? delivery-performance options))))))

